<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Exception\JsonRpc;

/**
 * Class AbstractJsonRpcException
 *
 * @package idoit\Module\Api\Exception\JsonRpc
 */
abstract class AbstractJsonRpcException extends \Exception
{
    /**
     * Error code constants
     */
    const PARSE            = -32700;
    const INVALID_REQUEST  = -32600;
    const METHOD_NOT_FOUND = -32601;
    const INVALID_PARAMS   = -32602;
    const INTERNAL_ERROR   = -32603;
    const SYSTEM_ERROR     = -32099;

    /**
     * Exception Register
     *
     * @var array
     */
    public static $register = [
        self::PARSE            => ParseException::class,
        self::INVALID_REQUEST  => RequestException::class,
        self::METHOD_NOT_FOUND => MethodException::class,
        self::INVALID_PARAMS   => ParameterException::class,
        self::INTERNAL_ERROR   => InternalErrorException::class,
        self::SYSTEM_ERROR     => SystemException::class,
    ];

    /**
     * Get Exception by code
     *
     * @param $errorCode
     *
     * @return string
     */
    public static function getExceptionByCode($errorCode)
    {
        if (!isset(self::$register[$errorCode])) {
            return SystemException::class;
        }

        return self::$register[$errorCode];
    }

    /**
     * Error related data
     *
     * @var array
     */
    protected $data = null;

    /**
     * Get error code
     *
     * @return int
     */
    public abstract function getErrorCode();

    /**
     * Get error topic
     *
     * @return string
     */
    public abstract function getErrorTopic();

    /**
     * Get data
     *
     * @return array
     */
    public function getData()
    {
        return $this->data;
    }

    /**
     * Set data
     *
     * @param array $data
     *
     * @return AbstractJsonRpcException
     */
    public function setData(array $data)
    {
        $this->data = $data;

        return $this;
    }

    /**
     * AbstractJsonRpcException constructor.
     *
     * @param string          $message
     * @param int             $code
     * @param \Throwable|null $previous
     */
    public function __construct($message = "", $code = 0, \Throwable $previous = null)
    {
        parent::__construct($message, $code, $previous);
    }
}