<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_dialog extends isys_api_model_cmdb implements isys_api_model_interface
{

    /**
     * Mode-Constants
     */
    const CL__DIALOG__UPDATE = 'update';
    const CL__DIALOG__CREATE = 'create';
    const CL__DIALOG__DELETE = 'delete';

    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => [
            "property",
            "catgID",
            "catsID",
            'category_id', // @see API-427 some dialogs are dependent to the category entry id
        ]
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [
        'read'   => [
            'property',
        ],
        'create' => [
            'property',
            'value',
        ],
        'update' => [
            'property',
            'value',
            'entry_id',
        ],
        'delete' => [
            'property',
            'entry_id'
        ],
    ];

    /**
     * Retrieve the category dao by given Parameter.
     *
     * @param   array          $p_params
     * @param   isys_cmdb_dao  $p_dao
     *
     * @return  bool|isys_cmdb_dao_category
     * @throws  isys_exception_api
     * @throws  isys_exception_database
     */
    private static function get_dao_class($p_params, $p_dao = null)
    {
        $l_dao = (!empty($p_dao) ? $p_dao : new isys_cmdb_dao(isys_application::instance()->container->get('database')));

        // Process Parameters.
        if (@$p_params[C__CMDB__GET__CATS]) {
            $l_get_param  = C__CMDB__GET__CATS;
            $l_cat_suffix = 's';
        } elseif (@$p_params[C__CMDB__GET__CATG]) {
            $l_get_param  = C__CMDB__GET__CATG;
            $l_cat_suffix = 'g';
        } elseif (isset($p_params['category']) && is_string($p_params['category'])) {
            if (strpos($p_params['category'], 'C__CATG__CUSTOM_FIELDS_') !== false && defined($p_params['category'])) {
                $dao = new isys_cmdb_dao_category_g_custom_fields($l_dao->get_database_component());

                $dao->set_catg_custom_id(constant($p_params['category']));

                return $dao;
            }

            if (strstr($p_params['category'], 'C__CATG')) {
                $l_cat_suffix = 'g';
            } else {
                $l_cat_suffix = 's';
            }

            $l_get_param = 'category';
        } else {
            throw new isys_exception_api(
                'Category ID missing. You must specify the parameter \'catsID\' or \'catgID\' ' .
                'in order to identify the corresponding category you would like to read data from.',
                -32602
            );
        }

        // Get category info.
        if (is_numeric(addslashes($p_params[$l_get_param]))) {
            $l_isysgui = $l_dao->get_isysgui('isysgui_cat' . $l_cat_suffix, (int) $p_params[$l_get_param])
                ->__to_array();
        } else {
            $l_isysgui = $l_dao->get_isysgui('isysgui_cat' . $l_cat_suffix, null, null, addslashes($p_params[$l_get_param]))
                ->__to_array();
        }

        // Check class and instantiate it.
        if (class_exists($l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"])) {
            /* Process data */
            if (($l_cat = new $l_isysgui["isysgui_cat{$l_cat_suffix}__class_name"]($l_dao->get_database_component()))) {
                return $l_cat;
            }
        }

        return false;
    }

    /**
     * Read category data.
     *
     * Parameter structure:
     *    [
     *       'property' => 'manufacturer' | array('title', 'manufacturer),
     *       'category' => 'C__CATG__CPU'
     *    ]
     *
     * @param array $params
     *
     * @return array|bool
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     */
    public function read($params)
    {
        $properties = $params["property"];

        if (!is_array($properties)) {
            $properties = [$properties];
        }

        /** @var isys_cmdb_dao_category|false $categoryDao */
        $categoryDao = self::get_dao_class($params);

        // Process data.
        if (!$categoryDao) {
            throw new isys_exception_api('The provided category could not be found!');
        }

        /**
         * Get translation object
         */
        $language = isys_application::instance()->container->get('language');

        $daoProperties = $categoryDao->get_properties();
        $response      = [];

        if (!is_array($daoProperties) || count($properties) === 0) {
            throw new isys_exception_api("You have to deliver at least one property parameter.");
        }

        $categoryEntryId = $params['category_id'] ?? null;
        $request = null;

        if ($categoryEntryId) {
            $request = new isys_request();
            $request->set_category_data_id($categoryEntryId);
        }

        foreach ($properties as $property) {
            if (!isset($daoProperties[$property])) {
                continue;
            }

            $properties = $daoProperties[$property];

            if ($properties["ui"]["type"] != 'dialog' && $properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_strPopupType'] != "dialog_plus") {
                continue;
            }

            // @see  API-41
            if ($categoryDao instanceof isys_cmdb_dao_category_g_custom_fields) {
                $identifier = $properties['ui']['params']['identifier'];

                $l_dao = new isys_cmdb_dao_dialog_admin($this->m_dao->get_database_component());
                $table = 'isys_dialog_plus_custom';

                if ($this->useAuth) {
                    isys_auth_dialog_admin::instance()
                        ->custom(isys_auth::VIEW, $identifier);
                }

                $l_res = $l_dao->get_data($table, null, 'isys_dialog_plus_custom__identifier = ' . $l_dao->convert_sql_text($identifier));
            } else {
                if (isset($properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'])) {
                    $propertyArData = $properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['p_arData'];
                    if ($propertyArData instanceof isys_callback) {
                        $l_res = $propertyArData->execute($request);
                    } elseif (is_array($propertyArData)) {
                        $l_res = $propertyArData;
                    }
                    $data = [];
                    foreach ($l_res as $id => $stringValue) {
                        if (stripos($stringValue, 'LC_') === 0) {
                            $title = $language->get($stringValue);
                        } else {
                            $title = $stringValue;
                        }
                        $data[] = [
                            'id' => $id,
                            'const' => null,
                            'title' => $title,
                        ];
                    }
                    return $data;
                }

                $condition = null;

                if (!isset($params['ignoreCondition']) || !$params['ignoreCondition']) {
                    if (isset($properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["condition"])) {
                        $condition = $properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["condition"];
                    }
                }

                $l_dao = new isys_cmdb_dao_dialog_admin($this->m_dao->get_database_component());
                $table = $properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strTable"];

                // @see API-451 Provide a proper message if the property is not fully defined.
                if (trim($table) === '') {
                    throw new isys_exception_api('It is not possible to read the dialog data of the selected property.');
                }

                // @see API-451 Provide a proper message if the table does not exist.
                if (!$l_dao->table_exists($table)) {
                    throw new isys_exception_api("The table '{$table}' of the selected property does not exist in your database.");
                }

                if ($this->useAuth) {
                    isys_auth_dialog_admin::instance()
                        ->table(isys_auth::VIEW, $table);
                }

                $l_res = $l_dao->get_data($properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strTable"], null, $condition);
            }

            if ($l_res->num_rows()) {
                /**
                 * @see API-32 Consider parent table
                 */
                $parentTable = $l_dao->get_parent_table($properties[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strTable"]);

                while ($l_row = $l_res->get_row()) {
                    // Setup general data values
                    $data = [
                        "id"    => $l_row[$table . "__id"],
                        "const" => isset($l_row[$table . "__const"]) ? $l_row[$table . "__const"] : '',
                        "title" => _L($l_row[$table . "__title"]),
                    ];

                    // Handle parentTable relation of entry if needed
                    if ($parentTable) {
                        $data['parent'] = [
                            'id'    => $l_row[$parentTable . '__id'],
                            'const' => isset($l_row[$$parentTable . "__const"]) ? $l_row[$parentTable . "__const"] : '',
                            'title' => _L($l_row[$parentTable . '__title'])
                        ];
                    }

                    // Persist entry in response
                    $response[] = $data;
                }
            }
        }

        return $response;
    }

    /**
     * Create Dialog+ Entry
     * Parameter structure:
     *    [
     *       'property'   => 'manufacturer',
     *       'cat(g|s)ID' => 'C__CATG__CPU',
     *       'value'      => 'New Dialog+ Entry Title'
     *    ]
     *
     * @param array $p_params
     *
     * @return array
     * @throws Exception
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function create($p_params)
    {
        return $this->dialog_routine($p_params, self::CL__DIALOG__CREATE);
    }

    /**
     * Delete Dialog+ Entry
     * Parameter structure:
     *    [
     *       'property'   => 'manufacturer',
     *       'cat(g|s)ID' => 'C__CATG__CPU',
     *       'entry_id'   => DIALOG_ENTRY_ID
     *    ]
     *
     *
     * @param array $p_params
     *
     * @return array
     * @throws Exception
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function delete($p_params)
    {
        return $this->dialog_routine($p_params, self::CL__DIALOG__DELETE);
    }

    /**
     * Update Dialog+ Entry.
     * Parameter structure:
     *    [
     *       'property'   => 'manufacturer',
     *       'cat(g|s)ID' => 'C__CATG__CPU',
     *       'value'      => 'New Dialog+ Entry Title',
     *       'entry_id'   => DIALOG_ENTRY_ID
     *    ]
     *
     * @param array $p_params
     *
     * @return array
     * @throws Exception
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     * @author Selcuk Kekec <skekec@i-doit.com>
     */
    public function update($p_params)
    {
        return $this->dialog_routine($p_params, self::CL__DIALOG__UPDATE);
    }

    /**
     * Create/Update/Delete Dialog+
     *
     * @author Selcuk Kekec <skekec@i-doit.com>
     *
     * @param array  $p_params
     * @param string $p_mode CL__DIALOG__[CREATE|UPDATE|DELETE]
     *
     * @return array
     * @throws Exception
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_database
     */
    private function dialog_routine($p_params, $p_mode = self::CL__DIALOG__CREATE)
    {
        // Response Array.
        $l_response = ["success" => false];

        // Check: Category [G/S] identifier.
        if ((isset($p_params[C__CMDB__GET__CATG]) && !empty($p_params[C__CMDB__GET__CATG])) ||
            (isset($p_params[C__CMDB__GET__CATS]) && !empty($p_params[C__CMDB__GET__CATS])) ||
            (isset($p_params['category']) && !empty($p_params['category']))) {
            // Retrieve Category-DAO.
            if (!($l_cat = self::get_dao_class($p_params, $this->m_dao))) {
                throw new isys_exception_api("Unable to retrieve category DAO by given identifier");
            }

            // SET value to dummy.
            if ($p_mode == self::CL__DIALOG__DELETE) {
                $p_params["value"] = "DUMMY";
            }

            // Parameter: Property and Value are setted and not empty.
            // @see API-411 Don't use 'empty()' for checking values.
            if (!(isset($p_params["property"]) && !empty($p_params["property"]) && isset($p_params["value"]) && trim($p_params["value"]) !== '')) {
                throw new isys_exception_api("Required Parameter is not set or empty. Please be sure that 'property' and 'value' are set.");
            }

            // Check for id.
            if ($p_mode == self::CL__DIALOG__UPDATE && (!isset($p_params['entry_id']) || empty($p_params['entry_id']))) {
                throw new isys_exception_api("Please specify an Entry-ID.");
            }

            $l_properties = $l_cat->get_properties();

            // Does Property exists in category?
            if (!(is_string($p_params["property"]) && isset($l_properties[$p_params["property"]]))) {
                throw new isys_exception_api("Property '" . $p_params["property"] . "' does not exist in Category");
            }

            // Is Property a Dialog+
            if (!(isset($l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strPopupType"]) &&
                $l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strPopupType"] == 'dialog_plus')) {
                throw new isys_exception_api("Property is not a Dialog+.");
            }

            // Initialize Dialog-Admin.
            $l_dao      = new isys_cmdb_dao_dialog_admin($this->m_dao->get_database_component());
            $identifier = null;
            $l_strTable = null;

            if (isset($l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strTable"])) {
                $l_strTable = $l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_strTable"];
            }

            // @see  API-258  Prioritize the defined source table instead of the UI Param.
            if (isset($l_properties[$p_params["property"]][C__PROPERTY__DATA][C__PROPERTY__DATA__SOURCE_TABLE])) {
                $l_strTable = $l_properties[$p_params["property"]][C__PROPERTY__DATA][C__PROPERTY__DATA__SOURCE_TABLE];
            }

            if ($l_cat instanceof isys_cmdb_dao_category_g_custom_fields) {
                $l_strTable = 'isys_dialog_plus_custom';
                $identifier = isset($l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_identifier"])
                    ? $l_properties[$p_params["property"]][C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]["p_identifier"]
                    : null;
            }

            /**
             * @see API-32 Handling parentId if needed
             */
            $parent = null;

            // Check whether parent relation is present and value was provided by request
            $parentTable = $l_dao->get_parent_table($l_strTable);
            if ($parentTable && !empty($p_params['parent'])) {
                $parent = $p_params['parent'];

                // Check whether provided value is not numric
                if (is_string($parent)) {
                    // Try to translate title to its id
                    $parent = $l_dao->get_by_title($parentTable, $parent)
                        ->get_row_value($parentTable . '__id');

                    if (!is_numeric($parent)) {
                        throw new Exception('Parent value \'' . $p_params['parent'] . '\' does not exist in ' . $parentTable . '. Please provide a value that is already present in parent table.');
                    }
                    $p_params['parent'] = (int)$parent;
                } elseif (!$l_dao->get_data($parentTable, $parent)->count()) {
                    throw new Exception('Unable to find corresponding entry for id \'' . $parent . '\' in parent table.');
                }
            }

            // Create / Update.
            switch ($p_mode) {
                case self::CL__DIALOG__CREATE:
                    if ($this->useAuth) {
                        $auth = isys_auth_dialog_admin::instance();

                        if ($l_strTable === 'isys_dialog_plus_custom') {
                            $auth->custom(isys_auth::CREATE, $identifier);
                        } else {
                            $auth->table(isys_auth::CREATE, $l_strTable);
                        }
                    }

                    // Create new Dialog-Entry.
                    if (!($l_intID = $l_dao->create($l_strTable, $p_params["value"], null, null, C__RECORD_STATUS__NORMAL, ($p_params['parent'] ?: null), $identifier))) {
                        throw new isys_exception_api("Unable to create a dialog entry.");
                    }
                    break;
                case self::CL__DIALOG__UPDATE:
                    if ($this->useAuth) {
                        isys_auth_dialog_admin::instance()->table(isys_auth::EDIT, $l_strTable);
                    }

                    if (!($l_intID = $l_dao->save($p_params["entry_id"], $l_strTable, $p_params["value"], null, null, C__RECORD_STATUS__NORMAL, ($p_params['parent'] ?: null)))) {
                        throw new isys_exception_api("Unable to update dialog entry.");
                    }

                    $l_intID = $p_params["entry_id"];
                    break;
                case self::CL__DIALOG__DELETE:
                    if ($this->useAuth) {
                        isys_auth_dialog_admin::instance()->table(isys_auth::DELETE, $l_strTable);
                    }

                    if (!($l_intID = $l_dao->delete($l_strTable, $p_params["entry_id"]))) {
                        throw new isys_exception_api("Unable to delete dialog entry.");
                    }

                    $l_intID = $p_params["entry_id"];
                    break;
            }

            // Set Response array.
            $l_response["entry_id"] = $l_intID;
            $l_response["success"]  = true;
        } else {
            throw new isys_exception_api("Given category identifier is not valid or not setted.");
        }

        return $l_response;
    }

    /**
     * isys_api_model_cmdb_dialog constructor.
     *
     * @param isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}
