<?php

use idoit\Component\Helper\ArrayHelper;
use idoit\Module\Api\CmdbObject;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\ValidationException;

/**
 * i-doit
 *
 * API model.
 *
 * @package     i-doit
 * @subpackage  API
 * @author      Dennis Stücken <dstuecken@synetics.de>
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_api_model_cmdb_object extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods.
     *
     * @var  array
     */
    protected $m_mapping = [
        'isys_obj__id'                     => 'id',
        'isys_obj__title'                  => 'title',
        'isys_obj__sysid'                  => 'sysid',
        'isys_obj__isys_obj_type__id'      => 'objecttype',
        'isys_obj_type__title'             => ['_L', 'type_title'],
        'isys_obj_type__icon'              => 'type_icon',
        'isys_obj__status'                 => 'status',
        'isys_obj__isys_cmdb_status__id'   => 'cmdb_status',
        'isys_cmdb_status__title'          => ['_L', 'cmdb_status_title'],
        'isys_obj__created'                => 'created',
        'isys_obj__updated'                => 'updated',
        'isys_catg_image_list__image_link' => ['isys_api_model_cmdb_objects::get_image_url', 'image']
    ];

    /**
     * Possible options and their parameters.
     *
     * @var  array
     */
    protected $m_options = [
        'read'       => [
            'id' => [
                'type'        => 'int',
                'description' => 'Object id',
                'reference'   => 'isys_obj__id',
                'optional'    => false,
            ],
        ],
        'create'     => [
            'title'       => [
                'type'        => 'string',
                'description' => 'Object title',
                'reference'   => 'isys_obj__title',
                'optional'    => false,
            ],
            'type'        => [
                'type'        => 'int|string',
                'description' => 'Object type as string constant or id',
                'reference'   => 'isys_obj_type__id',
                'optional'    => false,
            ],
            'cmdb_status' => [
                'type'        => 'int|string',
                'description' => 'Cmdb status id or constant',
                'reference'   => 'isys_obj__isys_cmdb_status__id',
                'optional'    => true,
            ],
        ],
        'update'     => [
            'id' => [
                'type'        => 'int',
                'description' => 'Object id',
                'reference'   => 'isys_obj__id',
                'optional'    => false,
            ],
        ],
        'delete'     => [
            'id'     => [
                'type'        => 'int',
                'description' => 'Object id',
                'reference'   => 'isys_obj__id',
                'optional'    => false,
            ],
            'status' => [
                'type'        => 'int',
                'description' => 'Object status',
                'reference'   => 'isys_obj__status',
                'optional'    => true,
            ],
        ],
        'quickpurge' => [
            'id' => [
                'type'        => 'int',
                'description' => 'Object id',
                'reference'   => 'isys_obj__id',
                'optional'    => false,
            ],
        ],
    ];

    /**
     * Validation.
     *
     * @var  array
     */
    protected $m_validation = [];

    /**
     * Fetches information about an object.
     *
     * @param   array $p_params Parameters. Structure: array('id' => 1).
     *
     * @return  array  Returns an empty array when an error occures.
     */
    public function read($p_params)
    {
        $l_return = [];

        if (isset($p_params[C__CMDB__GET__OBJECT])) {
            $p_params['id'] = $p_params[C__CMDB__GET__OBJECT];
            unset($p_params[C__CMDB__GET__OBJECT]);
        }

        if (isset($p_params['id']) && $p_params['id']) {
            $this->m_log->info('Retrieving object with id ' . $p_params['id']);

            if ($this->useAuth) {
                isys_auth_cmdb::instance()
                    ->check_rights_obj_and_category(isys_auth::VIEW, $p_params['id'], 'C__CATG__GLOBAL');
            }

            // Data retrieval.
            $l_data = $this->m_dao->get_object_by_id($p_params['id']);

            // Data formatting.
            if ($l_data->count() > 0) {
                return $this->format_by_mapping($this->m_mapping, $l_data->get_row());
            }
        } else {
            $this->m_log->error('Object ID missing.');
        }

        return $l_return;
    }
    /**
     * Validate object data.
     *
     * @param array $data
     *
     * @return array|null
     */
    public function validate_global_data($data)
    {
        return (new isys_cmdb_dao_category_g_global($this->get_database()))->validate($data);
    }

    /**
     * Creates an object.
     *
     * @param array $p_params
     *
     * @return array|null
     * @throws ValidationException
     * @throws isys_exception_api
     * @throws isys_exception_auth
     * @throws isys_exception_dao
     * @throws isys_exception_database
     * @throws isys_exception_general
     * @throws isys_exception_template
     */
    public function create($p_params)
    {
        // List of affected categories
        $affectedCategories = ['C__CATG__GLOBAL'];

        $l_return = [];

        if (!isset($p_params['title'])) {
            throw new isys_exception_api('Object title missing');
        }

        if (!isset($p_params['type']) || empty($p_params['type']) || !$p_params['type']) {
            throw new isys_exception_api('Object type missing');
        }

        $allowedStatuses = [
            C__RECORD_STATUS__NORMAL,
            C__RECORD_STATUS__ARCHIVED,
            C__RECORD_STATUS__DELETED,
            C__RECORD_STATUS__MASS_CHANGES_TEMPLATE,
            C__RECORD_STATUS__TEMPLATE
        ];

        // Validate status value
        if (!empty($p_params['status']) && (!is_int($p_params['status']) || !in_array($p_params['status'], $allowedStatuses, false))) {
            throw new ValidationException('status', 'Unable to handle status value.');
        }

        // Check for a provided SYS-ID.
        $sysid = null;

        if (isset($p_params['sysid']) && !empty($p_params['sysid'])) {
            $sysid = $p_params['sysid'];
        }

        if (is_array($validatedData = $this->validate_global_data($p_params))) {
            $errors = '';
            foreach ($validatedData as $key => $value) {
                $errors .= $key . ' => ' . $value . ', ';
            }
            throw new ValidationException('Validation errors: ', $errors);
        }

        // Check for a provided CMDB-Status.
        $cmdbStatus = null;

        if (isset($p_params['cmdb_status']) && !empty($p_params['cmdb_status'])) {
            if (is_numeric($p_params['cmdb_status'])) {
                $cmdbStatus = $p_params['cmdb_status'];
            } elseif (defined($p_params['cmdb_status'])) {
                $cmdbStatus = constant($p_params['cmdb_status']);
            }
        }

        try {
            // Object-Type.
            $objectType = null;
            $p_params['type'] = is_numeric($p_params['type']) ? $p_params['type'] : (defined($p_params['type']) ? constant($p_params['type']) : null);

            if ($p_params['type']) {
                $objectType = $this->m_dao->get_objtype($p_params['type'])->get_row();
            }

            $cat = new isys_cmdb_dao_category_g_global(isys_application::instance()->container->get('database'));
            $validation = $cat->validate($p_params);
            if (is_array($validation) && count($validation) > 0) {
                $l_validation_errors = '';
                foreach ($validation as $l_field => $l_problem) {
                    $l_validation_errors .= $l_field . ':' .  $l_problem . ', ';
                } // foreach

                throw new isys_exception_api_validation('There was a validation error: ' . rtrim($l_validation_errors, ', '), $validation);
            }

            if (!empty($objectType)) {
                if ($this->useAuth) {
                    isys_auth_cmdb::instance()->obj_in_type(isys_auth::CREATE, strtolower($objectType['isys_obj_type__const']));
                }

                // insert the object.
                $l_return['id'] = $this->m_dao->insert_new_obj(
                    $p_params['type'],
                    false,
                    $p_params['title'],
                    $sysid,
                    ($p_params['status'] ?: C__RECORD_STATUS__NORMAL),
                    null,
                    null,
                    false,
                    null,
                    null,
                    null,
                    null,
                    $p_params['category'],
                    $p_params['purpose'],
                    $cmdbStatus,
                    $p_params['description']
                );

                if (isset($p_params['defaultTemplate']) && $p_params['defaultTemplate']) {
                    // Get template module.
                    $l_template_module = new isys_module_templates();
                    $l_default_template = $this->m_dao->get_default_template_by_obj_type($p_params['type']);

                    $l_template_module->create_from_template(
                        [$l_default_template],
                        $p_params['type'],
                        $p_params['title'],
                        $l_return['id'],
                        false,
                        1,
                        '',
                        $p_params['category'],
                        $p_params['purpose']
                    );
                }

                if ($l_return['id'] > 0) {
                    // @see API-15 Force INTEGER type case.
                    $l_return['id'] = (int)$l_return['id'];

                    // Create logbook entry.
                    isys_event_manager::getInstance()
                        ->triggerCMDBEvent('C__LOGBOOK_EVENT__OBJECT_CREATED', '-object initialized-', $l_return['id'], $this->m_dao->get_objTypeID($l_return['id']));

                    $l_return['message'] = 'Object was successfully created';

                    // Check whether categories parameter is set
                    if (isset($p_params['categories'])) {
                        // Create object type dao
                        $objecTypeDao = new isys_cmdb_dao_object_type($this->get_database());
                        $categoryModel = new isys_api_model_cmdb_category($this->m_dao);
                        $categoryReturn = [];
                        $objectId = $l_return['id'];

                        // Iterate over categories to handle
                        foreach ($p_params['categories'] as $categoryConstant => $categoryEntries) {
                            // Check whether object type owns category
                            if (!$objecTypeDao->has_cat($p_params['type'], $categoryConstant)) {
                                $this->m_log->warning('It appears that the category "' . $categoryConstant . '" is not assigned to the object type "' . $objectType['isys_obj_type__const'] . '".');
                            }

                            // Add category to the list of affected categories
                            $affectedCategories[] = $categoryConstant;

                            // Iterate over category entries to handle
                            foreach ($categoryEntries as $categoryEntry) {
                                try {
                                    // Check whether category entry is not an array
                                    if (!is_array($categoryEntry)) {
                                        throw new Exception('Category data should be an array of key value pairs representating properties.');
                                    }

                                    // Create category entry
                                    $modelResult = $categoryModel->create([
                                        'objID'    => $objectId,
                                        'category' => $categoryConstant,
                                        'data'     => $categoryEntry
                                    ]);

                                    // Extract new category entry id
                                    $categoryReturn[$categoryConstant][] = $modelResult['id'];
                                } catch (isys_exception_api_validation $e) {
                                    // Build string representation of validation errors
                                    $validationErrors = array_map(function ($value, $key) {
                                        return $key . ': ' . $value;
                                    }, $e->get_validation_errors(), array_keys($e->get_validation_errors()));

                                    $categoryReturn[$categoryConstant][] = $e->getMessage() . ': ' . implode(';', $validationErrors);
                                } catch (Exception $e) {
                                    $categoryReturn[$categoryConstant][] = $e->getMessage();
                                }
                            }
                        }

                        $l_return['categories'] = $categoryReturn;
                    }

                    // Register object and categories for indexing
                    \idoit\Module\Api\SearchIndexRegister::register($l_return['id'], $affectedCategories);

                    $l_return['success'] = true;
                } else {
                    $l_return['message'] = 'Error while creating object';
                    $l_return['success'] = false;
                }
            } else {
                throw new isys_exception_api('Object type not found.');
            }
        } catch (isys_exception_cmdb $e) {
            throw new isys_exception_api($e->getMessage());
        }

        return $l_return;
    }

    /**
     * Updates data.
     *
     * @param array $p_params
     *
     * @return array|null
     * @throws isys_exception_api
     * @throws isys_exception_auth
     */
    public function update($p_params)
    {
        $l_return = [];

        if (!isset($p_params['id'])) {
            throw new isys_exception_api('Object id missing');
        }

        if (!isset($p_params['title'])) {
            throw new isys_exception_api('Please specify a new object title by setting \'title\' parameter.');
        }

        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::EDIT, $p_params['id']);
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$p_params['id']);

        $l_dao_global = new isys_cmdb_dao_category_g_global($this->m_dao->get_database_component());

        if ($l_dao_global->save_title($p_params['id'], $p_params['title'])) {
            // Create logbook entry.
            // @todo type should be retrieved by object ID.
            isys_event_manager::getInstance()->triggerCMDBEvent(
                'C__LOGBOOK_EVENT__CATEGORY_CHANGED',
                '- Object title changed to ' . $p_params['title'] . ' -',
                $p_params['id'],
                $this->m_dao->get_objTypeID($p_params['id']),
                'LC__CMDB__CATG__GLOBAL'
            );

            $l_dao_global->object_changed($p_params['id']);
            $l_return['message'] = 'Object title was successfully updated';
            $l_return['success'] = true;
        } else {
            $l_return['message'] = 'Error while updating object';
            $l_return['success'] = false;
        }

        // @see  API-191  Unlock the object once the request has finished.
        $this->unlockObject((int)$p_params['id']);

        return $l_return;
    }

    /**
     * Deletes an object.
     *
     * @param array $p_params Parameters (depends on data method).
     *
     * @return array
     * @throws isys_exception_api
     * @deprecated Please use 'purge' instead of this method.
     */
    public function quickpurge($p_params): array
    {
        $objectIds = ArrayHelper::flatten([$p_params['id'] ?? null, $p_params['ids'] ?? null]);

        // filter array for non-numeric values.
        $objectIds = array_unique(array_filter($objectIds, fn ($id) => is_numeric($id) && $id > 1));

        if (count($objectIds) === 0) {
            throw new isys_exception_api('Object id missing');
        }

        try {
            $skipped = [];

            foreach ($objectIds as $objectId) {
                try {
                    if ($this->useAuth) {
                        isys_auth_cmdb::instance()->obj_id(isys_auth::SUPERVISOR, $objectId);
                    }

                    $this->purge(['object' => $objectId]);
                } catch (Exception $e) {
                    $skipped[] = '#' . $objectId;

                    continue;
                }
            }

            $skippedMessage = count($skipped) ? ' But skipped ' . implode(', ', $skipped) : '';

            return [
                'message' => "Object(s) successfully purged!{$skippedMessage} [This method is deprecated and will be removed in a feature release. Use 'cmdb.object.purge' instead.]",
                'success' => true
            ];
        } catch (Throwable $e) {
            return [
                'message' => 'Error while quickpurging object(s) - ' . $e->getMessage(),
                'success' => false,
            ];
        }
    }

    /**
     * Recycle object
     *
     * @param array $params
     *
     * @return array
     * @throws Exception
     */
    public function recycle(array $params)
    {
        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::EDIT, $params['object']);
        }

        $object = new CmdbObject($params['object']);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        $sourceStatus = $object->getObjectStatus();

        if (in_array($sourceStatus, [C__RECORD_STATUS__TEMPLATE, C__RECORD_STATUS__MASS_CHANGES_TEMPLATE], false) && $this->m_dao->set_object_status($params['object'], C__RECORD_STATUS__NORMAL)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been recycled.'
            ];
        }

        if ($object->rank(C__RECORD_STATUS__NORMAL)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been recycled.'
            ];
        }

        throw new InternalErrorException('Unable to recycle object.');
    }

    /**
     * Archive object
     *
     * @param array $params
     *
     * @return array
     * @throws Exception
     */
    public function archive(array $params)
    {
        $object = new CmdbObject($params['object']);

        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::ARCHIVE, $params['object']);
        }

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        if ($object->rank(C__RECORD_STATUS__ARCHIVED)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been archived.'
            ];
        }

        throw new InternalErrorException('Unable to archive object.');
    }

    /**
     * Delete object
     *
     * @param array $params
     *
     * @return array
     * @throws Exception
     */
    public function delete($params)
    {
        // API-89 Legacy handling of cmdb.object.delete
        if (isset($params['id'], $params['status'])) {
            $methodMapping = [
                'C__RECORD_STATUS__ARCHIVED' => 'archive',
                'C__RECORD_STATUS__ARCHIVE'  => 'archive',
                'C__RECORD_STATUS__DELETED'  => 'delete',
                'C__RECORD_STATUS__PURGE'    => 'purge'
            ];

            $params['object'] = (int)$params['id'];

            unset($params['id']);

            // Check whether provided status is valid
            if (!$methodMapping[$params['status']]) {
                throw new Exception('The status parameter value is invalid - it should be one of: "' . implode('", "', array_keys($methodMapping)) . '".');
            }

            return $this->{$methodMapping[$params['status']]}($params);
        }

        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::DELETE, $params['object']);
        }

        $object = new CmdbObject($params['object']);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        if ($object->rank(C__RECORD_STATUS__DELETED)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been deleted.'
            ];
        }

        throw new InternalErrorException('Unable to delete object.');
    }

    /**
     * Purge object
     *
     * @param array $params
     *
     * @return array
     * @throws Exception
     */
    public function purge(array $params): array
    {
        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::SUPERVISOR, $params['object']);
        }

        $object = new CmdbObject($params['object']);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        if ($object->purge()) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been purged.'
            ];
        }

        throw new InternalErrorException('Unable to purge object.');
    }

    /**
     * Mark as mass change template
     *
     * @param array $params
     *
     * @return array
     * @throws InternalErrorException
     * @throws Exception
     */
    public function markAsMassChangeTemplate(array $params)
    {
        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::EDIT, $params['object']);
        }

        $object = new CmdbObject($params['object']);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        if ($object->rank(C__RECORD_STATUS__MASS_CHANGES_TEMPLATE)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been marked as mass change template.'
            ];
        }

        throw new InternalErrorException('Unable to mark object as mass change template.');
    }

    /**
     * Mark as template
     *
     * @param array $params
     *
     * @return array
     * @throws InternalErrorException
     * @throws Exception
     */
    public function markAsTemplate(array $params)
    {
        if ($this->useAuth) {
            isys_auth_cmdb::instance()->obj_id(isys_auth::EDIT, $params['object']);
        }

        $object = new CmdbObject($params['object']);

        // @see  API-191  Check and react to the object lock.
        $this->checkObjectLock((int)$params['object']);

        if ($object->rank(C__RECORD_STATUS__TEMPLATE)) {
            // @see  API-191  Unlock the object once the request has finished.
            $this->unlockObject((int)$params['object']);

            return [
                'success' => true,
                'message' => 'Object ' . $params['object'] . ' has been marked as template.'
            ];
        }

        throw new InternalErrorException('Unable to mark object as template.');
    }

    /**
     * Format by mapping
     *
     * @param array $p_mapping
     * @param array $p_row
     *
     * @return array
     */
    protected function format_by_mapping(array $p_mapping, $p_row)
    {
        // Get mapped result
        $result = parent::format_by_mapping($p_mapping, $p_row);

        // Convert numeric values to its integer representation
        $result['id'] = (int)$result['id'];
        $result['status'] = (int)$result['status'];
        $result['cmdb_status'] = (int)$result['cmdb_status'];
        $result['objecttype'] = (int)$result['objecttype'];

        return $result;
    }

    /**
     * Constructor.
     *
     * @param  isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;

        parent::__construct();
    }
}
