<?php

use Carbon\Carbon;
use idoit\Module\Api\Exception\JsonRpc\AuthenticationException;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\License\LicenseService;
use idoit\Module\License\LicenseServiceFactory;

/**
 * i-doit APi
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_idoit_license implements isys_api_model_interface
{
    /**
     * Read license information
     *
     * @param array $params
     *
     * @return array
     * @throws AuthenticationException
     * @throws InternalErrorException
     */
    public function read($params)
    {
        global $g_license_token;

        /** @var isys_component_session $session */
        $session = isys_application::instance()->container->get('session');

        // Check for an valid session first
        if (!$session->is_logged_in()) {
            throw new AuthenticationException('Unable to retrieve license information without valid and active user session. Please provide an api key first.');
        }

        // Check whether license module exists
        if (!class_exists('isys_module_licence')) {
            throw new InternalErrorException('Unable to instantiate license module.');
        }

        // Authenticate user
        if (class_exists('isys_auth_system_licence')) {
            isys_auth_system_licence::instance()
                ->overview(isys_auth::EXECUTE);
        }

        // Retrieve the current tenant ID.
        $tenantId = (int)$session->get_mandator_id();

        // Create a new license service instance.
        $licenseService = LicenseServiceFactory::createDefaultLicenseService(isys_application::instance()->container->get('database_system'), $g_license_token);

        // Check our two license types.
        $licenseEntities = $licenseService->getLicenses();
        $oldLicenses = $licenseService->getLegacyLicenses();

        if (empty($licenseEntities) && empty($oldLicenses)) {
            return null;
        }

        $tenant = $licenseService->getTenants(true, [isys_application::instance()->session->get_mandator_id()]);

        $usedObjects = 0;

        try {
            $tenantDatabase = $licenseService->retrieveConnectionForTenant($tenant[0]);
            $statisticsDao = new isys_statistics_dao($tenantDatabase, isys_cmdb_dao::instance($tenantDatabase));
            $usedObjects += $statisticsDao->count_objects();
        } catch (\Exception $exception) {
            // silent
        }

        unset($tenantDatabase, $statisticsDao);

        $licenseResult = [
            'objectCapacity' => [
                'total' => (int) $tenant[0]['isys_mandator__license_objects'],
                'inUse' => $usedObjects
            ],
            'addons' => $licenseService->getLicensedAddOns(),
            'licenses' => []
        ];

        foreach ($licenseEntities as $id => $licenseEntity) {
            $start = \Carbon\Carbon::createFromTimestamp($licenseEntity->getValidityFrom()->getTimestamp());
            $end = \Carbon\Carbon::createFromTimestamp($licenseEntity->getValidityTo()->getTimestamp());

            $invalid = (\Carbon\Carbon::now()->between($start, $end));

            $start = $start->format('Y-m-d H:i:s');
            $end = $end->format('Y-m-d H:i:s');

            $licenseResult['licenses'][] = [
                'id' => $id,
                'label' => $licenseEntity->getProductName() ?: $licenseEntity->getProductIdentifier(),
                'licenseType' => $licenseEntity->getProductType(),
                'registrationDate' =>  $start,
                'validUntil' => $end,
                'objects' => $licenseEntity->getObjects(),
                'tenants' => $licenseEntity->getTenants(),
                'environment' => $licenseEntity->getEnvironment(),
                'valid' => $invalid
            ];
        }

        foreach ($oldLicenses as $oldLicense) {
            $start = Carbon::createFromTimestamp($oldLicense[LicenseService::C__LICENCE__REG_DATE]);
            $end = Carbon::parse($oldLicense[LicenseService::LEGACY_LICENSE_EXPIRES]);

            $invalid = (\Carbon\Carbon::now()->between($start, $end));

            $start = $start->format('Y-m-d H:i:s');
            $end = $end->format('Y-m-d H:i:s');

            $label = 'Subscription (Classic)';
            $tenants = 1;

            if (in_array(
                $oldLicense[LicenseService::LEGACY_LICENSE_TYPE],
                LicenseService::LEGACY_LICENSE_TYPES_HOSTING,
                false
            )) {
                $label = 'Hosting (Classic)';
                $tenants = 50;
            }

            $licenseResult['licenses'][] = [
                'id' => (int) $oldLicense[LicenseService::LEGACY_LICENSE_ID],
                'label' => $label,
                'licenseType' => $label,
                'registrationDate' =>  $start,
                'validUntil' => $end,
                'objects' => $oldLicense[LicenseService::C__LICENCE__OBJECT_COUNT],
                'tenants' => $tenants,
                'environment' => 'production',
                'valid' => $invalid
            ];
        }

        return $licenseResult;
    }
}
