<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Category;

use idoit\Module\Api\Property\PropertyReadEntityDefinition;
use idoit\Module\Api\Property\PropertyWriteEntityDefinition;

/**
 * Class Descriptor
 *
 * @package idoit\Module\Api\Category
 */
class Descriptor
{
    /**
     * Category dao instance
     *
     * @var \isys_cmdb_dao_category
     */
    protected $daoInstance;

    /**
     * Raw category descriptor data
     *
     * @var array
     */
    protected $descriptor;

    /**
     * Get category descriptor by constant
     *
     * @param $constant
     *
     * @return Descriptor
     * @throws \Exception
     */
    public static function byConstant($constant)
    {
        // Get dao instance
        $categoryDescriptor = \isys_application::instance()->container->get('cmdb_dao')
            ->get_cat_by_const($constant);

        // Check descriptor is empty
        if (empty($categoryDescriptor)) {
            throw new \Exception('Unable to retrieve category description by given constant');
        }

        // Check descriptor is other than an array
        if (!is_array($categoryDescriptor)) {
            throw new \Exception('Category descriptor has to be array.');
        }

        return new self($categoryDescriptor);
    }

    /**
     * Get category descriptor by id
     *
     * @param int $categoryType
     * @param int $categoryId
     *
     * @return Descriptor
     */
    public static function byId($categoryType, $categoryId)
    {
        if (!in_array($categoryType, ['g', 's', 'g_custom'])) {
            throw new \Exception('Category type is unknown. Please use one of \'g\', \'s\' or \'g_custom\'.');
        }

        // Get constant of category by its id and type
        $categoryConstant = \isys_application::instance()->container->get('cmdb_dao')
            ->get_cat_const_by_id($categoryId, $categoryType);

        if (empty($categoryConstant || !is_string($categoryConstant))) {
            throw new \Exception('Category with given id and type does not exist.');
        }

        return self::byConstant($categoryConstant);
    }

    /**
     * Get category dao instance
     *
     * @return \isys_cmdb_dao_category
     */
    public function getDaoInstance()
    {
        return $this->daoInstance;
    }

    /**
     * Get category raw descriptor
     *
     * @return array
     */
    public function getDescriptor()
    {
        return $this->descriptor;
    }

    /**
     * Get category id
     *
     * @return int
     */
    public function getId()
    {
        return (int)$this->getDescriptorParameter('id');
    }

    /**
     * Get category type
     *
     * @return int
     */
    public function getType()
    {
        return (int)$this->getDescriptorParameter('type');
    }

    /**
     * Get category title
     *
     * @return string
     */
    public function getTitle()
    {
        return $this->getDescriptorParameter('title');
    }

    /**
     * Get translated title
     *
     * @return string
     * @throws \Exception
     */
    public function getTranslatedTitle()
    {
        return \isys_application::instance()->container->get('language')
            ->get($this->getTitle());
    }

    /**
     * Get category description
     *
     * @return string
     */
    public function getDescription()
    {
        return $this->getDescriptorParameter('description');
    }

    /**
     * Get category constant
     *
     * @return string
     */
    public function getConstant()
    {
        return $this->getDescriptorParameter('const');
    }

    /**
     * Get category source table
     *
     * @return string
     */
    public function getSourceTable()
    {
        return $this->getDescriptorParameter('source_table');
    }

    /**
     * Get category dao class name
     *
     * @return string
     */
    public function getClassName()
    {
        return $this->getDescriptorParameter('class_name');
    }

    /**
     * Is category multi valued?
     *
     * @return bool
     */
    public function isMultivalue()
    {
        return !!(int)$this->getDescriptorParameter('list_multi_value');
    }

    /**
     * Is category global?
     *
     * @return bool
     */
    public function isGlobal()
    {
        return $this->getType() === C__CMDB__CATEGORY__TYPE_GLOBAL;
    }

    /**
     * Is category specific?
     *
     * @return bool
     */
    public function isSpecific()
    {
        return $this->getType() === C__CMDB__CATEGORY__TYPE_SPECIFIC;
    }

    /**
     * Is category custom?
     *
     * @return bool
     */
    public function isCustom()
    {
        return $this->getType() === C__CMDB__CATEGORY__TYPE_CUSTOM;
    }

    /**
     * Is category virtual?
     *
     * @return bool
     */
    public function isVirtual()
    {
        return $this->getDescriptorParameter('virtual');
    }

    /**
     * Generic descriptor parameter getter
     *
     * @param string $parameter
     *
     * @return mixed
     */
    protected function getDescriptorParameter($parameter)
    {
        return $this->descriptor[$parameter];
    }

    /**
     * Check whether category is viertual or not
     *
     * @param string $categoryConstant
     *
     * @return bool
     */
    public static function isCategoryVirtual($categoryConstant)
    {
        return \in_array($categoryConstant, [
            'C__CATG__CABLE_CONNECTION',
            'C__CATG__CABLING',
            'C__CATG__CLUSTER_SHARED_STORAGE',
            'C__CATG__CLUSTER_SHARED_VIRTUAL_SWITCH',
            'C__CATG__CLUSTER_VITALITY',
            'C__CATG__JDISC_DISCOVERY',
            'C__CATG__LIVESTATUS',
            'C__CATG__MULTIEDIT',
            'C__CATG__NDO',
            'C__CATG__NET_ZONE',
            'C__CATG__NET_ZONE_SCOPES',
            'C__CATG__OBJECT_VITALITY',
            'C__CATG__RACK_VIEW',
            'C__CATG__SANPOOL',
            'C__CATG__STACK_MEMBERSHIP',
            'C__CATG__STACK_PORT_OVERVIEW',
            'C__CATG__STORAGE',
            'C__CATG__VIRTUAL_AUTH',
            'C__CATG__VIRTUAL_SUPERNET',
            'C__CATG__VIRTUAL_TICKETS',
            'C__CATG__VRRP_VIEW',
            'C__CATS__BASIC_AUTH',
            'C__CATS__CHASSIS_CABLING',
            'C__CATS__PDU_OVERVIEW',
            'C__CATG__VIRTUAL',
            'C__CATG__DATABASE_FOLDER',
        ]);
    }

    /**
     * Get property definitions
     *
     * @return array
     * @throws \Exception
     */
    public function getDefinition()
    {
        $language = \isys_application::instance()->container->get('language');

        // Begin category definition
        $definition = [
            'id'         => (int)$this->getId(),
            'title'      => $this->getTranslatedTitle(),
            'multiValue' => (bool)$this->isMultivalue(),
            'constant'   => $this->getConstant(),
            'properties' => [],
            'virtual'    => false,
        ];

        // Get category dao instance
        $categoryDaoInstance = $this->getDaoInstance();

        // Check whether category is virtual
        if (self::isCategoryVirtual($categoryDaoInstance->get_category_const())) {
            // Mark it as virtual
            $definition['virtual'] = true;
        }

        // Get properties of category
        $properties = $categoryDaoInstance->get_properties();

        // Iterate over it and create its definition
        foreach ($properties as $propertyKey => $propertyDefinition) {
            // @see API-313 Exclude virtual properties.
            if (isset($propertyDefinition[C__PROPERTY__PROVIDES][C__PROPERTY__PROVIDES__VIRTUAL]) && $propertyDefinition[C__PROPERTY__PROVIDES][C__PROPERTY__PROVIDES__VIRTUAL]) {
                continue;
            }

            $property = [
                'title'        => $language->get($propertyDefinition[C__PROPERTY__INFO][C__PROPERTY__INFO__TITLE]),
                'key'          => $propertyKey,
                'type'         => $propertyDefinition[C__PROPERTY__DATA][C__PROPERTY__DATA__TYPE],
                'propertyType' => $propertyDefinition[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE],
                'entity'       => null,
                'field'        => $propertyDefinition[C__PROPERTY__DATA][C__PROPERTY__DATA__FIELD],
                'ref'          => $propertyDefinition[C__PROPERTY__DATA][C__PROPERTY__DATA__REFERENCES][1],
                'mandatory'    => $propertyDefinition[C__PROPERTY__CHECK][C__PROPERTY__CHECK__MANDATORY]
            ];

            // Get entity read definition
            $property['readEntity'] = PropertyReadEntityDefinition::get($categoryDaoInstance, $propertyKey, $propertyDefinition)->toArray();

            // get entity write definition
            $property['writeEntity'] = PropertyWriteEntityDefinition::get($categoryDaoInstance, $propertyKey, $propertyDefinition)->toArray();

            // Add it to properties
            $definition['properties'][] = $property;
        }

        return $definition;
    }

    /**
     * Descriptor constructor.
     *
     * @param array $categoryDescriptor
     *
     * @throws \Exception
     */
    public function __construct(array $categoryDescriptor)
    {
        // Check whether dao class exists
        if (!class_exists($categoryDescriptor['class_name'])) {
            throw new \Exception('Category dao ' . $categoryDescriptor['class_name'] . ' does not exist.');
        }

        // Create category instance
        $categoryDaoInstance = new $categoryDescriptor['class_name'](\isys_application::instance()->container->get('database'));

        // Check whether category inherits from category dao
        if (!($categoryDaoInstance instanceof \isys_cmdb_dao_category)) {
            throw new \Exception('Invalid dao instance for category. Expected dao class which inherits from isys_cmdb_dao_category.');
        }

        // @see  API-232  Set the custom category ID.
        if ($categoryDaoInstance instanceof \isys_cmdb_dao_category_g_custom_fields) {
            $categoryDaoInstance->set_catg_custom_id($categoryDescriptor['id']);
        }

        // Store information
        $this->daoInstance = $categoryDaoInstance;
        $this->descriptor = $categoryDescriptor;
    }
}
