<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use Exception;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_application;
use isys_cmdb_dao_category_g_version;

/**
 * SoftwareAssignmentProcessor
 *
 * @package    idoit\Module\Api\Model\Category
 */
class SoftwareAssignmentProcessor extends AbstractCategoryProcessor implements RequestModifier, SyncModifier
{

    /**
     * Raw request
     *
     * @var array
     */
    protected $rawRequest = [];

    /**
     * @return array
     */
    public function getAffectedPropertiesByRequest()
    {
        return [
            'assigned_version',
        ];
    }

    /**
     * @return array
     */
    public function getAffectedPropertiesBySync()
    {
        return [
            'assigned_database_schema',
            'assigned_it_service',
            'assigned_databases',
        ];
    }

    /**
     * Modify api request
     *
     * @param array $request
     *
     * @return array
     */
    public function modifyRequest(array $request)
    {
        $isCreationRequest = in_array($request['option'], ['save', 'update', 'create'], true);

        // @see ID-9329 Check if 'assigned_version' was passed and if it makes sense.
        if ($isCreationRequest && isset($request['data']['application']) && $request['data']['application'] > 0 && isset($request['data']['assigned_version'])) {
            $request['data']['assigned_version'] = $this->versionExists((int)$request['data']['application'], $request['data']['assigned_version']);
        }

        $this->rawRequest = $request;

        return $request;
    }

    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     */
    public function modifySyncData(array $syncData)
    {
        $keys = [
            'assigned_database_schema',
            'assigned_it_service',
            'assigned_databases',
        ];

        /**
         * Check whether assigned database schema was setted explicitly by request
         * or was added by isys_api_model_cmdb_category::__to_sync_structure().
         *
         * In last cas we will remove it because the key will simply hold the
         * isys_catg_relation__id but sync will interpret it as object id.
         */
        foreach ($keys as $key) {
            if (is_array($this->rawRequest['data']) && !array_key_exists($key, $this->rawRequest['data'])) {
                $syncData['properties'][$key] = null;
            }
        }

        return $syncData;
    }

    /**
     * @param int $applicationObjectId
     * @param int|string $version
     *
     * @return int|null
     * @throws Exception
     */
    private function versionExists(int $applicationObjectId, $version): ?int
    {
        $result = isys_cmdb_dao_category_g_version::instance(isys_application::instance()->container->get('database'))->get_data(null, $applicationObjectId);

        while ($row = $result->get_row()) {
            if (is_int($version) && $version == $row['isys_catg_version_list__id']) {
                return (int)$row['isys_catg_version_list__id'];
            }

            if (is_string($version) && $version === $row['isys_catg_version_list__title']) {
                return (int)$row['isys_catg_version_list__id'];
            }
        }

        return null;
    }
}
