<?php

/**
 * i-doit api module installer.
 *
 * @package     i-doit
 * @copyright   synetics GmbH
 * @license     http://www.i-doit.com/license
 */
class isys_module_api_install
{
    /**
     * Init method.
     *
     * @param isys_component_database $mandatorDb
     * @param isys_component_database $systemDb
     * @param integer                 $moduleId
     * @param string                  $type
     * @param integer                 $tenantId
     *
     * @return void
     * @throws isys_exception_dao
     */
    public static function init($mandatorDb, $systemDb, $moduleId, $type, $tenantId = null)
    {
        // Check whether tenantId is set:
        if (!empty($tenantId)) {
            if ($type === 'install') {
                self::install($mandatorDb, $systemDb, $moduleId, $type, $tenantId);
            }

            if ($type === 'update') {
                self::update($mandatorDb, $systemDb, $moduleId, $type, $tenantId);
            }
        }
    }

    /**
     * Update
     *
     * @param      $mandatorDb
     * @param      $systemDb
     * @param      $moduleId
     * @param      $type
     * @param null $tenantId
     *
     * @throws isys_exception_dao
     */
    public static function update($mandatorDb, $systemDb, $moduleId, $type, $tenantId = null)
    {
        // Create tenant settings dao
        $tenantSettingsDao = new isys_component_dao_tenant_settings($systemDb, $tenantId);
        $systemSettingsDao = new isys_component_dao_settings($systemDb);

        // Get tenant settings:
        $tenantSettings = $tenantSettingsDao->get_settings();
        $systemSettings = $systemSettingsDao->get_settings();

        if (!array_key_exists('api.validation', $tenantSettings)) {
            $tenantSettingsDao->set('api.validation', 1);
        }

        if (!array_key_exists('api.authenticated-users-only', $systemSettings)) {
            $systemSettingsDao->set('api.authenticated-users-only', 1);
        }

        if (!array_key_exists('api.status', $tenantSettings)) {
            $tenantSettingsDao->set('api.status', 1);
        }

        if (!array_key_exists('api.log-level', $tenantSettings)) {
            $tenantSettingsDao->set('api.log-level', 100);
        }

        $tenantSettingsDao->remove('api.use-auth');
        $tenantSettingsDao->remove('logging.system.api');

        // Create settings dao
        $systemSettingsDao->remove('api.use-auth');
        $systemSettingsDao->remove('logging.system.api');

        // @see API466 Update rights from 'SYSTEM' to 'API'.
        $dao = new isys_cmdb_dao($mandatorDb);
        $moduleId = $dao->convert_sql_id($moduleId);

        // After installing / updating the add-on, set rights if they did not yet exist.
        if ($dao->retrieve("SELECT * FROM isys_auth WHERE isys_auth__isys_module__id = {$moduleId};")->count() > 0) {
            return;
        }

        $sql = "SELECT isys_auth__isys_obj__id AS object, isys_auth__type AS type, isys_auth__path AS path
            FROM isys_auth
            WHERE isys_auth__isys_module__id = (SELECT isys_module__id FROM isys_module WHERE isys_module__const = 'C__MODULE__SYSTEM' LIMIT 1)
            AND isys_auth__path LIKE 'JSONRPCAPI%';";

        $queries = [];
        $result = $dao->retrieve($sql);

        while ($row = $result->get_row()) {
            $path = trim(strtoupper($row['path']));
            $read = $row['type'] & 1;
            $edit = $row['type'] & 2;

            if (in_array($path, ['JSONRPCAPI/GLOBAL', 'JSONRPCAPI/*'], true) && ($read || $edit)) {
                $queries[] = [
                    'object' => (int)$row['object'],
                    'path'   => 'CONFIGURATION',
                    'type'   => $read + $edit
                ];
            }

            if (in_array($path, ['JSONRPCAPI/OBJTYPE', 'JSONRPCAPI/*'], true) && $read) {
                $queries[] = [
                    'object' => (int)$row['object'],
                    'path' => 'CATEGORIES_ATTRIBUTES',
                    'type' => $read
                ];
            }
        }

        foreach ($queries as $query) {
            $sql = "INSERT INTO isys_auth SET
                    isys_auth__isys_obj__id = {$query['object']},
                    isys_auth__type = {$query['type']},
                    isys_auth__isys_module__id = {$moduleId},
                    isys_auth__path = '{$query['path']}',
                    isys_auth__status = 2;";

            $dao->update($sql);
            $dao->apply_update();
        }
    }

    /**
     * Install
     *
     * @param isys_component_database $mandatorDb
     * @param isys_component_database $systemDb
     * @param                         $moduleId
     * @param                         $type
     * @param null                    $tenantId
     *
     * @throws isys_exception_dao
     */
    public static function install($mandatorDb, $systemDb, $moduleId, $type, $tenantId = null)
    {
        // Create tenant settings dao
        $tenantSettingsDao = new isys_component_dao_tenant_settings($systemDb, $tenantId);

        // Get tenant settings:
        $tenantSettingsDao->get_settings();

        $tenantSettingsDao->set('api.validation', 1);
        $tenantSettingsDao->set('api.status', 1);
        $tenantSettingsDao->set('api.log-level', 100); // Set to warning
        $tenantSettingsDao->remove('api.use-auth');
        $tenantSettingsDao->remove('logging.system.api');

        // Create settings dao
        $systemSettingsDao = new isys_component_dao_settings($systemDb);
        $systemSettingsDao->set('api.authenticated-users-only', 1);
        $systemSettingsDao->remove('api.use-auth');
        $systemSettingsDao->remove('logging.system.api');

        // Check whether mandator has an api key
        $resource = $systemDb->query('SELECT isys_mandator__apikey FROM isys_mandator WHERE isys_mandator__id = ' . $systemDb->escape_string($tenantId) .
            ' AND (isys_mandator__apikey IS NULL OR isys_mandator__apikey = "")');

        if ($systemDb->num_rows($resource)) {
            // Create a new random apu key
            $apiKey = substr(base_convert(sha1(uniqid(mt_rand())), 16, 36), 0, 16);

            // Prepare update sql to insert new api key for mandator
            $sql = 'UPDATE isys_mandator SET isys_mandator__apikey = \'' . $systemDb->escape_string($apiKey) . '\' WHERE isys_mandator__id = ' .
                $systemDb->escape_string($tenantId);

            // Execute query
            $systemDb->query($sql) && $systemDb->commit();
        }

        // @see API466 Create rights.
        $dao = new isys_cmdb_dao($mandatorDb);
        $moduleId = $dao->convert_sql_id($moduleId);
        $viewAndEditRight = isys_auth::VIEW + isys_auth::EDIT;
        $viewRight = isys_auth::VIEW;

        $sql = "INSERT INTO isys_auth SET
            isys_auth__isys_obj__id = (SELECT isys_obj__id FROM isys_obj WHERE isys_obj__const = 'C__OBJ__PERSON_GROUP_ADMIN' LIMIT 1),
            isys_auth__type = {$viewAndEditRight},
            isys_auth__isys_module__id = {$moduleId},
            isys_auth__path = 'CONFIGURATION',
            isys_auth__status = 2;";

        $dao->update($sql);
        $dao->apply_update();

        $sql = "INSERT INTO isys_auth SET
            isys_auth__isys_obj__id = (SELECT isys_obj__id FROM isys_obj WHERE isys_obj__const = 'C__OBJ__PERSON_GROUP_ADMIN' LIMIT 1),
            isys_auth__type = {$viewRight},
            isys_auth__isys_module__id = {$moduleId},
            isys_auth__path = 'CATEGORIES_ATTRIBUTES',
            isys_auth__status = 2;";

        $dao->update($sql);
        $dao->apply_update();
    }
}
