<?php

namespace idoit\Module\Api\Controller;

use Exception;
use idoit\Module\Api\Category\Descriptor;
use isys_application;
use isys_cmdb_dao_category;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * i-doit AttributeDocumentation controller.
 *
 * @package   api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class AttributeDocumentationController
{
    /**
     * @param Request $request
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function getCategoryList(Request $request): JsonResponse
    {
        $categoryStore = [];
        $cmdbDao = isys_application::instance()->container->get('cmdb_dao');
        $language = isys_application::instance()->container->get('language');

        $categoryTypeAbbrTitles = [
            'g' => $language->get('LC__CMDB__GLOBAL_CATEGORIES'),
            's' => $language->get('LC__CMDB__SPECIFIC_CATEGORIES'),
            'g_custom' => $language->get('LC__CMDB__CUSTOM_CATEGORIES'),
        ];

        foreach (['g', 's', 'g_custom'] as $categoryTypeAbbr) {
            $categoryResource = $cmdbDao->get_isysgui("isysgui_cat{$categoryTypeAbbr}");

            // @see ID-8942 Initialize the category type.
            $categoryStore[$categoryTypeAbbrTitles[$categoryTypeAbbr]] = [];

            while ($categoryRow = $categoryResource->get_row()) {
                $parentTitle = '';
                $idField = "isysgui_cat{$categoryTypeAbbr}__id";
                $titleField = "isysgui_cat{$categoryTypeAbbr}__title";
                $parentField = "isysgui_cat{$categoryTypeAbbr}__parent";
                $typeField = "isysgui_cat{$categoryTypeAbbr}__type";

                // @see API-309 Skip 'view' categories
                if ($categoryRow[$typeField] == isys_cmdb_dao_category::TYPE_VIEW) {
                    continue;
                }

                // @see  API-196 ID-7042  Categories with "parents" need to display their names in order to make sense.
                if (isset($categoryRow[$parentField]) && $categoryRow[$parentField]) {
                    if ($categoryTypeAbbr === 'g') {
                        $parentTitle = $language->get($cmdbDao->get_catg_name_by_id_as_string($categoryRow[$parentField])) . ' > ';
                    } elseif ($categoryTypeAbbr === 's') {
                        // @see API-338 Use a different method to get 'parent' categories.
                        $parents = [];
                        $parentResult = $cmdbDao->cats_get_parent_cats($categoryRow[$idField], true);

                        while ($parentRow = $parentResult->get_row()) {
                            $parents[] = $language->get($parentRow['isysgui_cats__title']);
                        }

                        if (count($parents)) {
                            $parentTitle = implode(' / ', $parents) . ' > ';
                        }
                    }
                }

                $categoryStore[$categoryTypeAbbrTitles[$categoryTypeAbbr]][] = [
                    'value' => $categoryTypeAbbr . '.' . $categoryRow[$idField],
                    'title' => $parentTitle . $language->get($categoryRow[$titleField])
                ];
            }

            usort($categoryStore[$categoryTypeAbbrTitles[$categoryTypeAbbr]], function ($a, $b) {
                return strnatcasecmp($a['title'], $b['title']);
            });
        }

        return new JsonResponse([
            'success' => true,
            'data'    => $categoryStore,
            'message' => ''
        ]);
    }

    /**
     * @param Request $request
     *
     * @return JsonResponse
     * @throws Exception
     */
    public function getCategoryDefinition(Request $request): JsonResponse
    {
        return new JsonResponse([
            'success' => true,
            'data'    => Descriptor::byId($request->query->get('categoryType'), (int)$request->query->get('categoryId'))->getDefinition(),
            'message' => ''
        ]);
    }
}
