<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Object;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\CmdbObject;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use isys_auth;
use isys_module_cmdb;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object delete endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Delete extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.object.delete.v2', 'Will delete a singular object from i-doit', [
            new RequiredParameter('object', Parameter::TYPE_INTEGER, 'Numeric object ID', fn ($id) => $id > 0),
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        $objectId = $request->request->get('object');

        $object = new CmdbObject($objectId);

        isys_module_cmdb::getAuth()->obj_id(isys_auth::DELETE, $objectId);

        // @see API-191 Check and react to the object lock.
        $this->lockObject($objectId);

        if (!$object->rank(C__RECORD_STATUS__DELETED)) {
            throw new InternalErrorException('Unable to delete object.');
        }

        // @see API-191 Unlock the object once the request has finished.
        $this->unlockObject($objectId);

        return new JsonRpcResponse([
            'success' => true,
            'message' => "Object {$objectId} has been deleted."
        ]);
    }
}
