<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Object;

use Exception;
use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use isys_application;
use isys_auth;
use isys_module_cmdb;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object read endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.object.read.v2', 'Will read a singular object from i-doit, can include category information', [
                new RequiredParameter('object', Parameter::TYPE_INTEGER, 'Numeric object ID', fn ($id) => $id > 0),
                new OptionalParameter('categories', Parameter::TYPE_ARRAY, 'Array of category constants', function ($categories) {
                    foreach ($categories as $categoryConstant) {
                        if (!defined($categoryConstant)) {
                            throw new Exception("The category constant '{$categoryConstant}' is not defined in your i-doit installation.");
                        }
                    }

                    return true;
                }, [])
            ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        $objectId = $request->request->get('object');
        $categories = $request->request->get('categories', []);

        isys_module_cmdb::getAuth()->check_rights_obj_and_category(isys_auth::VIEW, $objectId, 'C__CATG__GLOBAL');

        // Data retrieval.
        $objectResult = isys_application::instance()->container->get('cmdb_dao')
            ->get_object_by_id($objectId);

        // Data formatting.
        if (count($objectResult) === 0) {
            throw new InternalErrorException('No object exists with ID ' . $objectId);
        }

        $objectRow = $objectResult->get_row();

        $routeGenerator = isys_application::instance()->container->get('route_generator');
        $language = isys_application::instance()->container->get('language');

        $responseData = [
            'id'                => (int)$objectRow['isys_obj__id'],
            'title'             => $objectRow['isys_obj__title'],
            'sys-id'            => $objectRow['isys_obj__sysid'],
            'type-id'           => (int)$objectRow['isys_obj_type__id'],
            'type-title'        => $language->get($objectRow['isys_obj_type__title']),
            // 'type-icon'         => $routeGenerator->generate('cmdb.object-type.icon', ['objectTypeId' => $objectRow['isys_obj_type__id']]),
            'status'            => $objectRow['isys_obj__status'],
            'cmdb-status-id'    => (int)$objectRow['isys_obj__isys_cmdb_status__id'],
            'cmdb-status-title' => $language->get($objectRow['isys_cmdb_status__title']),
            'created'           => $objectRow['isys_obj__created'],
            'updated'           => $objectRow['isys_obj__updated'],
            // 'image'             => $routeGenerator->generate('cmdb.object.image', ['objectId' => $objectId])
        ];

        if (count($categories)) {
            $responseData['categories'] = [
                // @todo Read category data and apply it to the response data.
            ];
        }

        return new JsonRpcResponse($responseData);
    }
}
