<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Report;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Report\Report;
use idoit\Module\Report\Worker\ArrayWorker;
use isys_auth;
use isys_exception_api;
use isys_module_report;
use isys_report_dao;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB report read endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.report.read.v2', 'Reads a report.', [
            new RequiredParameter('id', Parameter::TYPE_INTEGER, 'The report ID', fn ($id) => $id > 0),
            new OptionalParameter('offset', Parameter::TYPE_INTEGER, 'Set a offset to skip the first n rows, can be used together with "limit" to page.', fn ($id) => $id > 0),
            new OptionalParameter('limit', Parameter::TYPE_INTEGER, 'Limit how many rows should be returned, can be used together with "offset" to page.', fn ($id) => $id > 0),
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        $id = $request->request->get('id');
        $offset = $request->request->get('offset', 0);
        $limit = $request->request->get('limit');

        $dao = isys_report_dao::instance();

        isys_module_report::getAuth()->custom_report(isys_auth::VIEW, $id);

        $report = $dao->get_reports(null, [$id], null, true, false);

        if (empty($report)) {
            throw new isys_exception_api("Report with id {$id} does not exist");
        }

        if (!$dao->hasQueryOnlyInternalPlaceholder($report[0]['isys_report__query'])) {
            throw new isys_exception_api("Report with id {$id} contains non internal placeholders, therefore cannot be executed by the api");
        }

        $query = $dao->replacePlaceHolders($report[0]['isys_report__query']);

        // @see API-370 Process the report with the 'ArrayWorker'.
        $worker = new ArrayWorker();
        $showHtml = (bool)$report[0]['isys_report__show_html'];
        // @see API-373 Check the 'compress multivalue results' option and set it later.
        $compressMultivalue = (bool)$report[0]['isys_report__compressed_multivalue_results'];

        (new Report($dao, $query))
            ->enableHtmlContext($showHtml)
            ->setCompressedMultivalueResults($compressMultivalue)
            ->setWorker($worker)
            ->execute();

        // @see API-428 Pass an empty array, if '$worker->export()' results in NULL.
        $responseData = array_slice($worker->export() ?? [], $offset, $limit);

        $removeScriptCallback = fn ($string) => $string;

        // @see API-370 Even though we want to see HTML we should remove 'script' tags.
        if ($showHtml) {
            // Took this line from 'Report->execute()' when HTML context is deactivated.
            $removeScriptCallback = fn ($string) => preg_replace("(<script[^>]*>([\\S\\s]*?)<\/script>)", '', $string);
        }

        foreach ($responseData as &$row) {
            foreach ($row as &$column) {
                // @see API-458 Replace '&nbsp;' with actual whitespaces and then trim the content.
                $column = trim(str_replace('&nbsp;', ' ', $removeScriptCallback($column)));
            }
        }

        return new JsonRpcResponse($responseData);
    }
}
