<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Dialog\DialogPlusAdmin;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;

/**
 * Class ModelProcessor
 *
 * @package idoit\Module\Api\Model\Cmdb\Category\Processor
 */
class StorProcessor extends AbstractCategoryProcessor implements RequestModifier
{
    /**
     * @return array
     */
    public function getAffectedPropertiesByRequest()
    {
        return [
            'manufacturer',
            'model',
        ];
    }

    /**
     * Modify api request
     *
     * @param array $request
     *
     * @return array
     * @throws ParameterException
     * @throws \Exception
     */
    public function modifyRequest(array $request)
    {
        // Check whether requested option is equal to read
        if ($request['option'] === 'read') {
            return $request;
        }

        // Initialize needed data
        $model = $request['data']['model'];
        $manufacturer = $request['data']['manufacturer'];
        $objectId = $request['objID'];
        $modelDao = \isys_cmdb_dao_category_g_model::instance(\isys_application::instance()->container->get('database'));
        $usingExistingManufacturer = false;

        // Check whether processing is necessary or not
        if (empty($model) && empty($manufacturer)) {
            return $request;
        }

        // Check whether creation is requested without setted manufacturer but model
        if ($request['option'] === 'create' && empty($manufacturer) && !empty($model)) {
            throw new ParameterException('Cannot set model without specified manufacturer.');
        }

        if (array_key_exists('manufacturer', $request['data']) && ($request['data']['manufacturer'] === null) && !empty($model)) {
            throw new ParameterException('Unable to set model during unsetting manufacturer parameter.');
        }

        // Create DialogAdmin
        $dialogAdmin = DialogPlusAdmin::byTableName('isys_stor_manufacturer', [
            C__PROPERTY__DATA => [C__PROPERTY__DATA__REFERENCES => ['isys_stor_manufacturer']]
        ]);

        // Handle manufacturer value
        if (is_string($manufacturer)) {
            // Create new manufacturer value
            $manufacturer = $dialogAdmin->findOrCreate($manufacturer);
        } else if (is_int($manufacturer)) {
            // Check whether given id exists
            if (!$dialogAdmin->idExists($manufacturer)) {
                $manufacturer = null;
            }
        } else if ($manufacturer === null) {
            // Get actual manufacturer
            $manufacturer = $modelDao->get_data(null, $objectId)->get_row_value('isys_catg_stor_list__isys_stor_manufacturer__id');

            $usingExistingManufacturer = true;
        }

        // Check whether manufacturer could be determined
        if (empty($manufacturer)) {
            throw new ParameterException('Unable to determine manufacturer');
        }

        // Handle model value
        if (is_string($model)) {
            // Check whether model exists
            $sql = 'SELECT isys_stor_model__id
                FROM isys_stor_model
                WHERE isys_stor_model__title = ' . $modelDao->convert_sql_text($model) . '
                AND isys_stor_model__isys_stor_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ';';

            $modelId = (int)$modelDao->retrieve($sql)->get_row_value('isys_stor_model__id');

            if ($modelId > 0) {
                // Use existing value
                $model = $modelId;
            } else {
                // Create new entry
                $sql = 'INSERT INTO isys_stor_model SET
                    isys_stor_model__title = ' . $modelDao->convert_sql_text($model) . ',
                    isys_stor_model__isys_stor_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ',
                    isys_stor_model__status = ' . $modelDao->convert_sql_int(C__RECORD_STATUS__NORMAL) . ';';

                if ($modelDao->update($sql) && $modelDao->apply_update()) {
                    $model = $modelDao->get_last_insert_id();
                }
            }
        } else if (is_int($model)) {
            $sql = 'SELECT isys_stor_model__id
                FROM isys_stor_model
                WHERE isys_stor_model__id = ' . $modelDao->convert_sql_id($model) . '
                AND isys_stor_model__isys_stor_manufacturer__id = ' . $modelDao->convert_sql_id($manufacturer) . ';';

            if (!$modelDao->retrieve($sql)->num_rows()) {
                $model = null;
            }
        } else {
            $model = null;
        }

        // Set calculated data
        if (!$usingExistingManufacturer) {
            $request['data']['manufacturer'] = $manufacturer;
        }

        // Set model
        $request['data']['model'] = $model;

        return $request;
    }
}
