<?php

use idoit\Module\Api\Category\Descriptor;

/**
 * i-doit
 *
 * API model for CMDB
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @author     Benjamin Heisig <bheisig@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_objects extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data mapping used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_obj__id'                     => 'id',
        'isys_obj__title'                  => 'title',
        'isys_obj__sysid'                  => 'sysid',
        'isys_obj__isys_obj_type__id'      => 'type',
        'isys_obj__created'                => 'created',
        'isys_obj__updated'                => 'updated',
        'isys_obj_type__title'             => ['_L', 'type_title'],
        'isys_obj_type__icon'              => 'type_icon',
        'isys_obj_type_group__id'          => 'type_group',
        'isys_obj_type_group__title'       => ['_L', 'type_group_title'],
        'isys_obj__status'                 => 'status',
        'isys_obj__isys_cmdb_status__id'   => 'cmdb_status',
        'isys_cmdb_status__title'          => ['_L', 'cmdb_status_title'],
        'isys_catg_image_list__image_link' => ['isys_api_model_cmdb_objects::get_image_url', 'image'],
        'categories'                       => 'categories'
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /** @var array */
    private array $categoryCache;

    /**
     * @param string $unused
     * @param array  $row
     *
     * @return string
     * @throws Exception
     */
    public static function get_image_url($unused,array $row): string
    {
        try {
            return isys_application::instance()->container->get('route_generator')->generate('cmdb.object.image', ['objectId' => $row['isys_obj__id']]);
        } catch (Throwable $e) {
            // @todo API-505 this is a workaround and should not be necessary, after the bootstrapping has been unified.
            return rtrim(isys_application::instance()->www_path, '/') . '/cmdb/object/image/' . ((int)$row['isys_obj__id']);
        }
    }

    /**
     * Fetches objects by filter.
     * [
     *    array    $p_params['filter']['ids']         (optional) Object identifiers
     *    integer  $p_params['filter']['type']        (optional) Object type
     *    integer  $p_params['filter']['type_group']  (optional) Object type group
     *    mixed    $p_params['filter']['status']      (optional) Record-Status
     *    string   $p_params['filter']['title']       (optional) Object title
     *    string   $p_params['filter']['sysid']       (optional) SYSID
     *    string   $p_params['filter']['first_name']  (optional) First name (person)
     *    string   $p_params['filter']['last_name']   (optional) Last name (person)
     *    string   $p_params['filter']['email']       (optional) Email address (person)
     *    integer  $p_params['filter']['location']    (optional) Location tree
     *    boolean  $p_params['raw']                   (optional) Formatting.
     *    string   $p_params['order_by']              (optional) Order by one of the supported filter arguments. Defaults to null that means result will be ordered by object identifiers.
     *    string   $p_params['sort']                  (optional) Order result ascending ('ASC') or descending ('DESC').
     *    integer  $p_params['limit']                 (optional) Limitation: where to start and number of elements, i.e. 0 or 0,10. Defaults to null that means no limitation.
     * ]
     *
     * @param array $p_params
     *
     * @return array Objects. Returns an empty array on error.
     * @author Benjamin Heisig <bheisig@synetics.de>
     */
    public function read($p_params)
    {
        if (!is_array($p_params)) {
            $p_params = [];
        }

        if (!is_array($p_params['filter']) || !isset($p_params['filter'])) {
            $p_params['filter'] = [];
        }

        // Check whether parameter `categories` is set
        if (isset($p_params['categories'])) {
            // Check whether it is valid
            if (!is_array($p_params['categories']) && !is_bool($p_params['categories'])) {
                throw new Exception('Categories needs to be an array of category constants or a boolean.');
            }

            // Check whether all constants are defined or not
            if (is_array($p_params['categories'])) {
                if (empty($p_params['categories'])) {
                    throw new Exception('Parameter \'categories\' is an array but does not include any category constants.');
                }

                foreach ($p_params['categories'] as $categoryConstant) {
                    if ((strpos($categoryConstant, 'C__CATG__') === 0 || strpos($categoryConstant, 'C__CATS__') === 0) && !defined($categoryConstant)) {
                        throw new Exception('Category constant in paramter \'categories\' does not exists');
                    }
                }
            }

            // Create category model once for performance reason
            $categoryModel = new isys_api_model_cmdb_category($this->m_dao);
        }

        // @see API-415 Unify the content of 'ids' filter.
        if (isset($p_params['filter']['ids']) && !is_array($p_params['filter']['ids'])) {
            if (is_numeric($p_params['filter']['ids'])) {
                $p_params['filter']['ids'] = [$p_params['filter']['ids']];
            } else {
                unset($p_params['filter']['ids']);
            }
        }

        // Force limit to record status 'normal':
        if (isset($p_params['filter']['status']) && is_string($p_params['filter']['status']) && defined($p_params['filter']['status'])) {
            $p_params['filter']['status'] = constant($p_params['filter']['status']);
        } else {
            $p_params['filter']['status'] = C__RECORD_STATUS__NORMAL;
        }

        $l_raw = (bool)$p_params['raw'];
        $l_order_by = null;
        $l_sort = null;
        $l_limit = null;

        // Order by:
        if (isset($p_params['order_by']) && !empty($p_params['order_by'])) {
            $l_order_by = $p_params['order_by'];
        }

        // Sort:
        if (isset($p_params['sort']) && !empty($p_params['sort'])) {
            $l_sort = $p_params['sort'];
        }

        // Limitation:
        if (isset($p_params['limit']) && !empty($p_params['limit'])) {
            $l_limit = $p_params['limit'];
        }

        $typeTitleFilter = null;
        if (isset($p_params['filter']['type_title'])) {
            $typeTitleFilter = $p_params['filter']['type_title'];
            unset($p_params['filter']['type_title']);
        }

        // Data retrieval:
        $l_data = $this->m_dao->get_objects($p_params['filter'], $l_order_by, $l_sort, $typeTitleFilter ? null : $l_limit);

        $l_return = [];

        $this->categoryCache = [];

        // Data formatting:
        while ($l_row = $l_data->get_row()) {
            if ($typeTitleFilter) {
                $translatedTypeTitle = $this->language->get($l_row['isys_obj_type__title']);
                if ($typeTitleFilter !== $translatedTypeTitle) {
                    continue;
                }
            }

            if ($this->useAuth) {
                try {
                    // @todo  Maybe better implement a auth SQL snippet in "$this->m_dao->get_objects(...)"?
                    isys_auth_cmdb::instance()->check_rights_obj_and_category(isys_auth::VIEW, $l_row['isys_obj__id'], 'C__CATG__GLOBAL');
                } catch (isys_exception_auth $e) {
                    $this->m_log->error($e->getMessage());
                    continue;
                }
            }

            // Check whether categories should be requested
            if (isset($p_params['categories'])) {
                // @see API-412 Don't filter requested categories - the API should return whatever was requested.
                $targetCategories = $p_params['categories'];
                // @see API-415 Re-introduce the 'getCategoryConstantsByObjectTypeId' method.
                if (is_bool($targetCategories)) {
                    $targetCategories = $this->getCategoryConstantsByObjectTypeId((int)$l_row['isys_obj__isys_obj_type__id']);
                }

                // Request categories for object
                $l_row['categories'] = $this->getCategoryData(
                    (int)$l_row['isys_obj__id'],
                    $targetCategories,
                    $categoryModel
                );
            }

            $l_return[] = ($l_raw ? $l_row : $this->format_by_mapping($this->m_mapping, $l_row));
        }

        if ($l_limit && $typeTitleFilter) {
            $l_return = array_slice($l_return, 0, $l_limit);
        }

        // Sort by translated titles:
        if ($l_sort === 'title' && $l_raw === false) {
            usort($l_return, [$this, 'sort_by_title']);
        }

        // Add location:
        if (isset($p_params['filter']['location']) && !empty($p_params['filter']['location'])) {
            $l_location_dao = new isys_cmdb_dao_category_g_location($this->m_db);
            $l_location_tree = $l_location_dao->get_location_tree()->__as_array();
            $l_id = $l_raw ? 'isys_obj__id' : $this->m_mapping['isys_obj__id'];

            $l_format_location = function ($l_entity_id) use (&$l_format_location, $l_location_tree) {
                foreach ($l_location_tree as $l_node) {
                    if ($l_node['id'] == $l_entity_id) {
                        return array_merge($l_format_location($l_node['parentid']), [$l_node['title']]);
                    }
                }

                return [];
            };

            foreach ($l_return as $l_index => $l_entity) {
                $l_return[$l_index]['location'] = array_slice($l_format_location($l_entity[$l_id]), 0, -1);
            }
        }

        return $l_return;
    }

    /**
     * Get category data for object by category constants
     *
     * @param int                          $objectId
     * @param array                        $categoryConstants
     * @param isys_api_model_cmdb_category $model
     *
     * @return array
     * @throws isys_exception_database
     * @throws Exception
     */
    private function getCategoryData(int $objectId, array $categoryConstants, isys_api_model_cmdb_category $model)
    {
        // Check whether object exists or not
        if (empty($objectId) || !$this->m_dao->obj_exists($objectId)) {
            throw new Exception('Unable to find object with id \'' . $objectId . '\’');
        }

        // Initilize data store
        $data = [];

        // Iterate over requested category constants
        foreach ($categoryConstants as $categoryConstant) {
            // Skip overview category
            if ($categoryConstant === 'C__CATG__OVERVIEW') {
                continue;
            }

            try {
                // Try to request category for given object
                $data[$categoryConstant] = $model->read(['objID' => $objectId, 'category' => $categoryConstant]);
            } catch (Exception $e) {
                // Exception occured - save it in data field
                $data[$categoryConstant][] = $e->getMessage();
            }
        }

        return $data;
    }

    /**
     * Get all category constants which are assigned to an object type
     * This method was removed during API-412 but reintroduced in API-415
     *
     * @param int $objectTypeId
     *
     * @return array
     * @throws isys_exception_database
     */
    public function getCategoryConstantsByObjectTypeId(int $objectTypeId)
    {
        if (!isset($this->categoryCache[$objectTypeId])) {
            // Category constant store
            $this->categoryCache[$objectTypeId] = [];

            // Retrieve global categories and store them
            $resource = $this->m_dao->get_catg_by_obj_type($objectTypeId);

            if (count($resource)) {
                while ($row = $resource->get_row()) {
                    // Check whether category is virtual
                    if (!$row['isysgui_catg__const'] || Descriptor::isCategoryVirtual($row['isysgui_catg__const'])) {
                        continue;
                    }

                    $this->categoryCache[$objectTypeId][] = $row['isysgui_catg__const'];
                }
            }

            // Retrieve specific categories and store them
            $resource = $this->m_dao->gui_get_cats_with_subcats_by_objtype_id($objectTypeId);

            if (count($resource)) {
                foreach ($resource as $category) {
                    if (!$category['isysgui_cats__const'] || Descriptor::isCategoryVirtual($category['isysgui_cats__const'])) {
                        continue;
                    }

                    $this->categoryCache[$objectTypeId][] = $category['isysgui_cats__const'];
                }
            }

            // Retrieve custom categories and store them
            $resource = $this->m_dao->get_catg_custom_by_obj_type($objectTypeId);

            if (count($resource)) {
                while ($row = $resource->get_row()) {
                    if (!$row['isysgui_catg_custom__const']) {
                        continue;
                    }
                    $this->categoryCache[$objectTypeId][] = $row['isysgui_catg_custom__const'];
                }
            }
        }

        return $this->categoryCache[$objectTypeId];
    }

    /**
     * Format by mapping
     *
     * @param array $p_mapping
     * @param array $p_row
     *
     * @return array
     */
    protected function format_by_mapping(array $p_mapping, $p_row)
    {
        // Get mapped result
        $result = parent::format_by_mapping($p_mapping, $p_row);

        // Convert numeric values to its integer representation
        $result['id'] = (int)$result['id'];
        $result['status'] = (int)$result['status'];
        $result['cmdb_status'] = (int)$result['cmdb_status'];
        $result['type'] = (int)$result['type'];

        return $result;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * Deletes an object.
     *
     * @param array $p_params Parameters (depends on data method)
     *
     * @return array
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        $l_return = [
            'message' => 'Error while deleting object(s)',
            'success' => false
        ];

        if (isset($p_params['ids'])) {
            $p_params['id'] = $p_params['ids'];
        }

        if (!isset($p_params['id'])) {
            throw new isys_exception_api('Object id(s) missing');
        }

        if (is_numeric($p_params['id'])) {
            $p_params['id'] = [$p_params['id']];
        }

        if (is_array($p_params['id'])) {
            foreach ($p_params['id'] as $l_id) {
                if ($this->useAuth) {
                    try {
                        isys_auth_cmdb::instance()->obj_id(isys_auth::DELETE, $l_id);
                    } catch (isys_exception_auth $e) {
                        $this->m_log->error($e->getMessage());
                        continue;
                    }
                }

                try {
                    // @see  API-191  Check and react to the object lock.
                    $this->checkObjectLock((int)$l_id);
                } catch (Exception $e) {
                    continue;
                }

                if (!$this->m_dao->set_object_status($l_id, C__RECORD_STATUS__DELETED)) {
                    throw new isys_exception_api(sprintf('Error while deleting object with id %s', $l_id));
                }

                // @see  API-191  Unlock the object once the request has finished.
                $this->unlockObject((int)$l_id);
            }

            $l_return['message'] = 'Object(s) successfully deleted';
            $l_return['success'] = true;
        }

        return $l_return;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * isys_api_model_cmdb_objects constructor.
     *
     * @param isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}
