<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

use idoit\Module\Api\ApiCommandRunner;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;

/**
 * Class isys_api_model_console
 */
class isys_api_model_console extends isys_api_model
{
    /**
     * Method mapping
     *
     * @var array
     */
    private static $methodMapping = [
        'tenant::list' => 'listTenants',
        'commands::list' => 'listCommands',
    ];

    /**
     * Route request
     *
     * @param string $method
     * @param array  $params
     *
     * @return $this|isys_api_model
     * @throws Exception
     */
    public function route($method, $params)
    {
        // Build model class
        $modelClassName = 'isys_api_model_console_' . $method;

        // Call data method and format data:
        if (class_exists($modelClassName)) {
            /**
             * @var isys_api_model_console
             */
            $modelInstance = new $modelClassName();
            // Extract target method
            $targetMethod = (array_key_exists($method . '::' . $params['option'], self::$methodMapping)) ? self::$methodMapping[$method . '::' . $params['option']] : $params['option'];

            // Check whether method exists in model
            if (method_exists($modelInstance, $targetMethod)) {
                $this->m_data = $modelInstance->$targetMethod($params);
            } else {
                throw new Exception('Method \'' . $targetMethod . '\' does not exists in model \'' . $modelClassName . '\'');
            }
        } else {
            throw new Exception('Model \'console.' . $method . '\' does not exist');
        }

        return $this;
    }

    /**
     * Build command runner
     *
     * @param string $commandName
     * @param array  $options
     * @param array  $arguments
     *
     * @return ApiCommandRunner
     * @throws Exception
     */
    protected function buildCommandRunner($commandName, $options, $arguments)
    {
        // Check whether command name is filled
        if (empty($commandName)) {
            throw new \Exception('Command name is mandatory. Please provide a valid string value that is referencing a command.');
        }

        // Create command runner
        $commandRunner = new ApiCommandRunner(isys_application::instance()->app_path, $commandName);

        // Check whether options are supplied
        if (is_array($options)) {
            // Add options to command runner
            foreach ($options as $name => $value) {
                $commandRunner->addOption($name, $value);
            }
        }

        // Check whether arguments are supplied
        if (is_array($arguments)) {
            // Add argument to command runner
            foreach ($arguments as $name => $value) {
                $commandRunner->addArgument($name, $value);
            }
        }

        return $commandRunner;
    }

    /**
     * Run command
     *
     * @param string $commandName
     * @param array  $options
     * @param array  $arguments
     *
     * @return array
     * @throws Exception
     */
    protected function run($commandName, $options, $arguments)
    {
        // Build command runner
        $commandRunner = $this->buildCommandRunner($commandName, $options, $arguments);

        // Run command
        $commandRunner->run();

        return [
            'success' => $commandRunner->success(),
            'output'  => $commandRunner->getCommandOutput()
        ];
    }

    /**
     * Check wether command exists or not
     *
     * @param string $commandClassName
     *
     * @return bool
     * @throws InternalErrorException
     */
    protected function commandExists($commandClassName)
    {
        // Check whether command exists
        if (!class_exists($commandClassName)) {
            throw new InternalErrorException('Unable to find desired command.');
        }

        return true;
    }
}
