<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Report;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Report\Report;
use idoit\Module\Report\Worker\ArrayWorker;
use isys_auth;
use isys_exception_api;
use isys_module_report;
use isys_report_dao;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB report read endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.report.read.v2', 'LC__ADDON__API__ENDPOINT__CMDB_REPORT_READ', [
            new RequiredParameter('id', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__ID', fn ($id) => $id > 0),
            new OptionalParameter('offset', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__REPORT_OFFSET', fn ($id) => $id > 0, 0),
            new OptionalParameter('limit', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__REPORT_LIMIT', fn ($id) => $id > 0),
        ], [
            'Basic request'                          => Helper::exampleRequest('cmdb.report.read.v2', ['id' => 123]),
            'Request with offset and limit (paging)' => Helper::exampleRequest('cmdb.report.read.v2', ['id' => 123, 'offset' => 150, 'limit' => 50])
        ], [
            'Example response' => [
                Helper::exampleResponse([
                    [
                        "Object type"     => "Client",
                        "Title"           => "Laptop 005",
                        "Order date"      => "31.01.2022",
                        "Delivery date"   => "04.01.2022",
                        "Date of invoice" => "01.02.2022"
                    ],
                    [
                        "Object type"     => "Client",
                        "Title"           => "Laptop 006",
                        "Order date"      => "03.03.2022",
                        "Delivery date"   => "04.02.2022",
                        "Date of invoice" => "01.03.2022"
                    ],
                    ['...']
                ])
            ]
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        $id = $request->request->get('id');
        $offset = $request->request->get('offset', 0);
        $limit = $request->request->get('limit');

        $dao = isys_report_dao::instance();

        isys_module_report::getAuth()->custom_report(isys_auth::VIEW, $id);

        $report = $dao->get_reports(null, [$id], null, true, false);

        if (empty($report)) {
            throw new isys_exception_api("Report with id {$id} does not exist");
        }

        if (!$dao->hasQueryOnlyInternalPlaceholder($report[0]['isys_report__query'])) {
            throw new isys_exception_api("Report with id {$id} contains non internal placeholders, therefore cannot be executed by the api");
        }

        $query = $dao->replacePlaceHolders($report[0]['isys_report__query']);

        // @see API-370 Process the report with the 'ArrayWorker'.
        $worker = new ArrayWorker();
        $showHtml = (bool)$report[0]['isys_report__show_html'];
        // @see API-373 Check the 'compress multivalue results' option and set it later.
        $compressMultivalue = (bool)$report[0]['isys_report__compressed_multivalue_results'];

        (new Report($dao, $query))
            ->enableHtmlContext($showHtml)
            ->setCompressedMultivalueResults($compressMultivalue)
            ->setWorker($worker)
            ->execute();

        // @see API-428 Pass an empty array, if '$worker->export()' results in NULL.
        $responseData = array_slice($worker->export() ?? [], $offset, $limit);

        $removeScriptCallback = fn ($string) => $string;

        // @see API-370 Even though we want to see HTML we should remove 'script' tags.
        if ($showHtml) {
            // Took this line from 'Report->execute()' when HTML context is deactivated.
            $removeScriptCallback = fn ($string) => preg_replace("(<script[^>]*>([\\S\\s]*?)<\/script>)", '', $string);
        }

        foreach ($responseData as &$row) {
            foreach ($row as &$column) {
                // @see API-458 Replace '&nbsp;' with actual whitespaces and then trim the content.
                $column = trim(str_replace('&nbsp;', ' ', $removeScriptCallback($column)));
            }
        }

        return new JsonRpcResponse($responseData);
    }
}
