<?php

namespace idoit\Module\Api\PushIdentifier;

use idoit\Module\Api\Category\Descriptor;

class PushIdentifier
{
    const OBJECT = 'OBJECT';

    /**
     * @var \isys_component_database
     */
    private \isys_component_database $database;

    /**
     * @var PushIdentifier
     */
    private static $instance = null;

    /**
     * @param \isys_component_database $database
     */
    public function __construct(\isys_component_database $database)
    {
        $this->database = $database;
    }

    /**
     * @param \isys_component_database $database
     *
     * @return PushIdentifier
     */
    public static function instance(\isys_component_database $database)
    {
        if (self::$instance !== null) {
            return self::$instance;
        }
        self::$instance = new self($database);

        return self::$instance;
    }

    /**
     * @return array
     */
    public function findExternalTypes(): array
    {
        $results = $this->database->query("SELECT DISTINCT isys_push_identifier__external_type FROM isys_push_identifier");

        $return = [];
        while ($row = $results->fetch_assoc()) {
            $return[] = $row['isys_push_identifier__external_type'];
        }

        return $return;
    }

    /**
     * @param string $externalType
     *
     * @return array
     * @throws \Exception
     */
    public function findExternalIds(string $externalType): array
    {
        $this->invalidateObjectIdentifier();
        $results = $this->database->query("SELECT isys_push_identifier__id, isys_push_identifier__external_id, isys_push_identifier__reference_type, isys_push_identifier__reference_id
FROM isys_push_identifier
WHERE isys_push_identifier__external_type = '{$this->database->escape_string($externalType)}'");

        $return = [];
        while ($row = $results->fetch_assoc()) {
            if (!$this->isValid($row['isys_push_identifier__reference_type'], $row['isys_push_identifier__reference_id'])) {
                $this->delete($row['isys_push_identifier__id']);
                continue;
            }
            $return[$row['isys_push_identifier__reference_id']] = $row['isys_push_identifier__external_id'];
        }

        return $return;
    }

    /**
     * @param string      $externalType
     * @param string|null $externalId
     *
     * @return array
     * @throws \Exception
     */
    public function findReferences(string $externalType, ?string $externalId = null): array
    {
        $this->invalidateObjectIdentifier();

        [$objectExtType, $objectExtId, $categoryConst] = explode('/', $externalType);
        $conditions = ["isys_push_identifier__external_type = '{$this->database->escape_string($externalType)}'"];

        if ($categoryConst === null && $externalId === null) {
            $conditions = ["isys_push_identifier__external_type LIKE '{$this->database->escape_string($externalType)}/%'"];
        }

        if ($externalId !== null) {
            $conditions[] = "isys_push_identifier__external_id = '{$this->database->escape_string($externalId)}'";
        }

        $results = $this->database->query(sprintf("SELECT
            isys_push_identifier__id,
            isys_push_identifier__reference_type,
            isys_push_identifier__external_type,
            isys_push_identifier__reference_id,
            isys_push_identifier__external_id,
            isys_push_identifier__created,
            isys_push_identifier__created_by,
            isys_push_identifier__updated,
            isys_push_identifier__updated_by
        FROM isys_push_identifier
        WHERE %s", implode(' AND ', $conditions)));

        $return = [];
        while ($row = $results->fetch_assoc()) {
            if (!$this->isValid($row['isys_push_identifier__reference_type'], $row['isys_push_identifier__reference_id'])) {
                $this->delete($row['isys_push_identifier__id']);
                continue;
            }
            $return[] = [
                'created'       => $row['isys_push_identifier__created'],
                'createdBy'     => $row['isys_push_identifier__created_by'],
                'updated'       => $row['isys_push_identifier__updated'],
                'updatedBy'     => $row['isys_push_identifier__updated_by'],
                'referenceType' => $row['isys_push_identifier__reference_type'],
                'referenceId'   => $row['isys_push_identifier__reference_id'],
                'id'            => $row['isys_push_identifier__id'],
                'externalType'  => $row['isys_push_identifier__external_type'],
                'externalId'    => $row['isys_push_identifier__external_id'],
            ];
        }

        return $return;
    }

    /**
     * @param string $externalType
     * @param string $externalId
     * @param string $referenceType
     * @param string $referenceId
     *
     * @return bool
     * @throws \Exception
     */
    public function createReference(string $externalType, string $externalId, string $referenceType, string $referenceId, string $createdBy): bool
    {
        if (!self::isObjectReference($referenceType)) {
            Descriptor::byConstant($referenceType);
        } elseif (!$this->isValid($referenceType, $referenceId)) {
            throw new \Exception("Unable to find category entry {$referenceId} in category {$referenceType}");
        }

        return $this->database->affected_rows($this->database->query("INSERT INTO isys_push_identifier SET
                 isys_push_identifier__external_type = '{$this->database->escape_string($externalType)}',
                 isys_push_identifier__external_id = '{$this->database->escape_string($externalId)}',
                 isys_push_identifier__reference_type = '{$this->database->escape_string($referenceType)}',
                 isys_push_identifier__reference_id = '{$this->database->escape_string($referenceId)}',
                 isys_push_identifier__created = NOW(),
                 isys_push_identifier__created_by = '{$this->database->escape_string($createdBy)}'") === 1);
    }

    /**
     * @param int    $id
     * @param string $updatedBy
     *
     * @return bool
     */
    public function touch(int $id, string $updatedBy): bool
    {
        return $this->database->affected_rows($this->database->query("UPDATE isys_push_identifier SET
                                isys_push_identifier__updated_by = '{$this->database->escape_string($updatedBy)}',
                                isys_push_identifier__updated = NOW()
                                WHERE isys_push_identifier__id = {$this->database->escape_string($id)}",) === 1);
    }

    /**
     * @param string $referenceType
     * @param int    $referenceId
     *
     * @return bool
     * @throws \Exception
     */
    private function isValid(string $referenceType, int $referenceId): bool
    {
        if (self::isObjectReference($referenceType)) {
            return true;
        }

        return Descriptor::byConstant($referenceType)
                ->getDaoInstance()
                ->get_data($referenceId)
                ->count() > 0;
    }

    /**
     * @param string $referenceType
     *
     * @return bool
     */
    public static function isObjectReference(string $referenceType): bool
    {
        return $referenceType === self::OBJECT;
    }

    /**
     * @param int $id
     *
     * @return bool
     */
    private function delete(int $id): bool
    {
        return $this->database->affected_rows($this->database->query("DELETE FROM isys_push_identifier WHERE isys_push_identifier__id = {$this->database->escape_string($id)}")) ===
            1;
    }

    /**
     * @param string $extType
     * @param array  $externalIds
     *
     * @return bool
     */
    public function deleteByExternalData(string $extType, array $externalIds): bool
    {
        $inCondition = implode(',', array_map(fn($item) => "'{$this->database->escape_string($item)}'", $externalIds));

        return $this->database->query("DELETE FROM isys_push_identifier
            WHERE isys_push_identifier__external_type = '{$this->database->escape_string($extType)}' AND
                  isys_push_identifier__external_id IN ({$inCondition})");
    }

    /**
     * Invalidate obsolete entries
     */
    private function invalidateObjectIdentifier(): void
    {
        $this->database->query("DELETE FROM isys_push_identifier
       WHERE isys_push_identifier__reference_type = '" . self::OBJECT . "' AND isys_push_identifier__reference_id NOT IN(SELECT isys_obj__id FROM isys_obj);");
    }

    /**
     * @param int $id
     *
     * @return bool
     */
    public function referenceExists(int $id): bool {
        return (int) $this->database->num_rows($this->database->query("SELECT 1 FROM isys_push_identifier WHERE isys_push_identifier__reference_id = '{$this->database->escape_string($id)}'
                                     and isys_push_identifier__reference_type = '{$this->database->escape_string(self::OBJECT)}'")) === 1;
    }

    /**
     * @param int    $id
     * @param string $referenceType
     *
     * @return array|null
     */
    public function getDataByReference(int $id, string $referenceType): ?array {
        $data = $this->database->fetch_array($this->database->query("SELECT * FROM isys_push_identifier WHERE isys_push_identifier__reference_id = '{$this->database->escape_string($id)}'
                                     and isys_push_identifier__reference_type = '{$this->database->escape_string($referenceType)}'"));
        return $data;
    }
}
