<?php /** @noinspection PhpIncludeInspection */

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api;

/**
 * Class ApiEnvironmentShell
 *
 * @package idoit\Module\Api
 */
class ApiEnvironmentShell
{

    /**
     * Boot error handler
     *
     * @var callable
     */
    private $bootErrorHandler;

    /**
     * Error handler
     *
     * @var callable
     */
    private $errorHandler;

    /**
     * Executable
     *
     * @var callable
     */
    private $executable;

    /**
     * i-doit root directory path
     *
     * @var string
     */
    private $idoitRootDirectoryPath;

    /**
     * Run shell
     *
     * This will run the executable and return its
     * value to its source. All errors will be redirected
     * to the handlers and should be handled there.
     *
     * @return bool
     * @throws \Exception
     */
    public function run()
    {
        // Check whether prerequirements are setted
        if (!$this->bootErrorHandler || !$this->errorHandler || !$this->executable) {
            throw new \Exception('ApiEnvironmentShell is not setted up. Unable to run executable!');
        }

        /**
         * Boot Sequence
         */
        try {
            // Boot api shell
            $this->boot();
        } catch (\Exception $e) {
            // Invoke boot error handler
            $this->invoke($this->bootErrorHandler, [$e]);

            return false;
        }

        /**
         * Execution Sequence
         */
        try {
            // Invoke executable
            return $this->invoke($this->executable, [$this]);
        } catch (\Exception $e) {
            // Invoke error handler
            $this->invoke($this->errorHandler, [$e]);
        }

        return false;
    }

    /**
     * Inject boot error handler
     *
     * @param callable $callable
     */
    public function injectBootErrorHandler(callable $callable)
    {
        $this->bootErrorHandler = $callable;
    }

    /**
     * Inject error handler
     *
     * @param callable $callable
     */
    public function injectErrorHandler(callable $callable)
    {
        $this->errorHandler = $callable;
    }

    /**
     * Inject executable to run
     *
     * @param callable $callable
     */
    public function injectExecutable(callable $callable)
    {
        $this->executable = $callable;
    }

    /**
     * Get idoit root directory path
     *
     * @return string
     */
    public function getIdoitRootDirectoryPath()
    {
        return $this->idoitRootDirectoryPath;
    }

    /**
     * Set idoit root directory path
     *
     * @param string $idoitRootDirectoryPath
     */
    private function setIdoitRootDirectoryPath($idoitRootDirectoryPath)
    {
        $this->idoitRootDirectoryPath = $idoitRootDirectoryPath;
    }

    /**
     * Boot api shell
     *
     * @throws \Exception
     */
    private function boot()
    {
        // @see API-544 It is necessary to fetch a list of variables from the 'outer' scope, even though we do not directly use them here.
        global $g_absdir, $g_db_system, $g_config, $g_product_info, $g_crypto_hash, $g_license_token, $g_is_cloud, $g_active_features;

        // Define web context constant
        define('WEB_CONTEXT', false);

        // Include minimal runtime environment if this script is called directly
        if (!isset($g_absdir)) {
            // Calculate error reporting
            $l_errorReporting = E_ALL & ~E_NOTICE;

            if (defined('E_DEPRECATED')) {
                $l_errorReporting &= ~E_DEPRECATED;
            }

            if (defined('E_WARNING')) {
                $l_errorReporting &= ~E_WARNING;
            }

            if (defined('E_STRICT')) {
                $l_errorReporting &= ~E_STRICT;
            }

            // Setup error reporting
            error_reporting($l_errorReporting);

            // Set absolute path to i-doit root directory
            $g_absdir = $this->idoitRootDirectoryPath;

            // Include bootstrapping and configuration files
            /** @noinspection PhpIncludeInspection */
            if (file_exists($this->idoitRootDirectoryPath . "/src/config.inc.php") && include_once($this->idoitRootDirectoryPath . "/src/config.inc.php")) {
                // Include global and caching environment.
                include_once($this->idoitRootDirectoryPath . "/src/bootstrap.inc.php");
                include_once($this->idoitRootDirectoryPath . "/src/caching.inc.php");
            }
        }

        // Run locales configuration
        if (!class_exists("isys_locale")) {
            require_once "locales.inc.php";
        }

        // Run api module initialization routine
        if (file_exists($this->idoitRootDirectoryPath . '/src/classes/modules/api/init.php')) {
            require_once $this->idoitRootDirectoryPath . '/src/classes/modules/api/init.php';
        }
    }

    /**
     * Invoke function and return its result
     *
     * @param callable $callable
     * @param array    $parameter
     *
     * @return mixed
     */
    private function invoke(callable $callable, array $parameter = [])
    {
        return call_user_func_array($callable, $parameter);
    }

    /**
     * ApiEnvironmentShell constructor.
     *
     * @param string $idoitRootDirectoryPath
     */
    public function __construct($idoitRootDirectoryPath)
    {
        $this->setIdoitRootDirectoryPath($idoitRootDirectoryPath);
    }
}
