<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Controller;

use idoit\Component\Logger;
use idoit\Module\Api\Request\RequestStore;
use isys_application;
use Monolog\Formatter\LineFormatter;
use Monolog\Handler\NullHandler;
use Monolog\Handler\RotatingFileHandler;
use Monolog\Handler\StreamHandler;

/**
 * Class AbstractController
 *
 * @package idoit\Module\Api\Controller
 */
abstract class AbstractController
{
    /**
     * Enable / Disable authentication system
     *
     * @var bool
     */
    protected $authenticationSystem = false;

    /**
     * Logger instance
     *
     * @var Logger
     */
    protected $logger;

    /**
     * RequestStore instance
     *
     * @var RequestStore
     */
    protected $requestStore;

    /**
     * Run request store
     *
     * @return mixed
     */
    public abstract function run();

    /**
     * Get logger instance
     *
     * @return Logger
     */
    public function getLogger()
    {
        return $this->logger;
    }

    /**
     * Set logger instance
     *
     * @param Logger $logger
     *
     * @return AbstractController
     */
    public function setLogger(Logger $logger)
    {
        $this->logger = $logger;

        return $this;
    }

    /**
     * Enable authentication system
     *
     * @return AbstractController
     */
    public function enableAuthenticationSystem()
    {
        return $this->setAuthenticationSystem(true);
    }

    /**
     * Disable authentication system
     *
     * @return AbstractController
     */
    public function disableAuthenticationSystem()
    {
        return $this->setAuthenticationSystem(false);
    }

    /**
     * @return bool
     */
    public function getAuthenticationSystem()
    {
        return $this->authenticationSystem;
    }

    /**
     * @param bool $authenticationSystem
     *
     * @return AbstractController
     */
    private function setAuthenticationSystem($authenticationSystem)
    {
        global $g_config;

        // Enable authenticat
        $g_config['use_auth'] = $authenticationSystem ? 1 : 0;

        $this->authenticationSystem = $authenticationSystem;

        return $this;
    }

    /**
     * Get request store
     *
     * @return RequestStore
     */
    public function getRequestStore()
    {
        return $this->requestStore;
    }

    /**
     * Set RequestStore
     *
     * @param RequestStore $requestStore
     *
     * @return AbstractController
     */
    public function setRequestStore($requestStore)
    {
        $this->requestStore = $requestStore;

        return $this;
    }

    /**
     * Create logger instance
     *
     * @return Logger
     * @throws \Exception
     */
    private function createLoggerInstance()
    {
        // Create and configure api logger instance
        $logger = new Logger('api.logger');
        // Get log level
        $logLevel = (int)\isys_tenantsettings::get('api.log-level', Logger::WARNING);

        if ($logLevel != -1) {
            // Create and setup line formatter
            $formatter = new LineFormatter("[%datetime%] %level_name%: %message% %context%\n", 'Y-m-d H:i:s u');
            $formatter->ignoreEmptyContextAndExtra(true);
            $formatter->allowInlineLineBreaks(true);

            if (method_exists($formatter, 'addJsonEncodeOption')) {
                $formatter->addJsonEncodeOption(JSON_PRETTY_PRINT);
            }

            $tenantId = (int)isys_application::instance()->container->get('session')->get_mandator_id();

            // Create and setup stream handler
            $streamHandler = new RotatingFileHandler(\isys_application::instance()->app_path . '/log/api_' . $tenantId . '.log', 0, $logLevel);
            $streamHandler->setFilenameFormat('{filename}_{date}', RotatingFileHandler::FILE_PER_DAY);
            $streamHandler->setFormatter($formatter);

            // Add file stream handler
            $logger->setHandlers([$streamHandler]);
        }

        return $logger;
    }

    /**
     * Get api version
     *
     * @return string
     */
    public function getApiVersion()
    {
        return (file_exists(__DIR__ . '/../../VERSION')) ? file_get_contents(__DIR__ . '/../../VERSION') : '0.0';
    }

    /**
     * AbstractController constructor.
     *
     * @throws \Exception
     */
    public function __construct()
    {
        // Create and setup logger instance
        $loggerInstance = $this->createLoggerInstance();

        // Set logger in controller context
        $this->setLogger($loggerInstance);
    }
}
