<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use Exception;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_application;
use isys_cmdb_dao_category_g_operating_system;
use isys_cmdb_dao_category_g_version;

/**
 * SoftwareAssignmentProcessor
 *
 * @package    idoit\Module\Api\Model\Category
 */
class SoftwareAssignmentProcessor extends AbstractCategoryProcessor implements RequestModifier, SyncModifier
{

    /**
     * Raw request
     *
     * @var array
     */
    protected $rawRequest = [];

    /**
     * @return array
     */
    public function getAffectedPropertiesByRequest()
    {
        return [
            'assigned_version',
        ];
    }

    /**
     * @return array
     */
    public function getAffectedPropertiesBySync()
    {
        return [
            'assigned_database_schema',
            'assigned_it_service',
            'assigned_databases',
        ];
    }

    /**
     * Modify api request
     *
     * @param array $request
     *
     * @return array
     * @throws Exception
     */
    public function modifyRequest(array $request)
    {
        $isCreationRequest = in_array($request['option'], ['save', 'update', 'create'], true);

        // @see ID-9329 Check if 'assigned_version' was passed and if it makes sense.
        if ($isCreationRequest && isset($request['data']['assigned_version'])) {
            $application = $this->findApplication((int)$request['object'], $request['data']);

            if ($application !== null) {
                $request['data']['assigned_version'] = $this->versionExists($application, $request['data']['assigned_version']);
            }
        }

        $this->rawRequest = $request;

        return $request;
    }

    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     */
    public function modifySyncData(array $syncData)
    {
        $keys = [
            'assigned_database_schema',
            'assigned_it_service',
            'assigned_databases',
        ];

        /**
         * Check whether assigned database schema was set explicitly by request
         * or was added by isys_api_model_cmdb_category::__to_sync_structure().
         *
         * In last case we will remove it because the key will simply hold the
         * isys_catg_relation__id but sync will interpret it as object id.
         */
        foreach ($keys as $key) {
            if (is_array($this->rawRequest['data']) && !array_key_exists($key, $this->rawRequest['data'])) {
                $syncData['properties'][$key] = null;
            }
        }

        return $syncData;
    }

    /**
     * Find the correct application in order to reference the version.
     *
     * @param int   $objectId
     * @param array $requestData
     *
     * @return int|null
     * @see API-513 Find correct application for later reference.
     */
    private function findApplication(int $objectId, array $requestData): ?int
    {
        if (isset($requestData['application']) && $requestData['application'] > 0) {
            return (int)$requestData['application'];
        }

        $application = null;

        if ($this->dao instanceof isys_cmdb_dao_category_g_operating_system) {
            // Fetch current application via host object.
            $application = $this->dao->get_data(null, $objectId)->get_row_value('isys_connection__isys_obj__id');
        } elseif (isset($requestData['id']) && $requestData['id'] > 0) {
            // Fetch current application via category entry.
            $application = $this->dao->get_data($requestData['id'])->get_row_value('isys_connection__isys_obj__id');
        }

        if ($application !== null) {
            return (int)$application;
        }

        return null;
    }

    /**
     * @param int $applicationObjectId
     * @param int|string $version
     *
     * @return int|null
     * @throws Exception
     */
    private function versionExists(int $applicationObjectId, $version): ?int
    {
        $versionDao = isys_cmdb_dao_category_g_version::instance(isys_application::instance()->container->get('database'));
        $result = $versionDao->get_data(null, $applicationObjectId);

        while ($row = $result->get_row()) {
            if (is_int($version) && $version == $row['isys_catg_version_list__id']) {
                return (int)$row['isys_catg_version_list__id'];
            }

            if (is_string($version) && $version === $row['isys_catg_version_list__title']) {
                return (int)$row['isys_catg_version_list__id'];
            }
        }

        // @see API-488 Create a new version on-the-fly.
        if (is_string($version)) {
            $newlyCreatedVersion = $versionDao->create($applicationObjectId, C__RECORD_STATUS__NORMAL, $version);

            if (is_numeric($newlyCreatedVersion)) {
                return (int)$newlyCreatedVersion;
            }
        }

        return null;
    }
}
