<?php

namespace idoit\Module\Api\Model\External;

use Exception;
use idoit\Module\Api\Category\Descriptor;
use idoit\Module\Cmdb\Component\CategoryLogger\CmdbLogger;
use idoit\Module\Cmdb\Component\CategoryLogger\CmdbRecord;
use idoit\Module\Api\Model\External\Push\Category;
use idoit\Module\Api\Model\External\Push\Entry;
use idoit\Module\Api\Model\External\Handler\CiObject;
use idoit\Module\Api\PushIdentifier\PushIdentifier;
use isys_application;
use isys_auth;
use isys_auth_cmdb;
use isys_auth_cmdb_categories;
use isys_cmdb_dao_category;

class Push extends AbstractExternal
{
    /**
     * @return Push
     * @throws Exception
     */
    public function prepare()
    {
        $title = $this->getConfig()->getTitle();
        $class = $this->getConfig()->getClass();
        $data = $this->getConfig()->getData();
        $extType = $this->getConfig()->getExtType();
        $extId = $this->getConfig()->getExtId();

        $logger = Logger::instance();
        $categoryLogger = CmdbLogger::instance();
        $language = isys_application::instance()->container->get('language');
        $strategyProvider = new CategoryStrategyProvider();
        $formattedData = [];

        if (strpos($extType, '/')) {
            throw new Exception("Unallowed symbol '/' is being used. Please remove or replace it.");
        }

        $ciObject = new CiObject($class, $title, $extType, $extId, PushIdentifier::OBJECT);
        $ciObject->find();

        try {
            if ($ciObject->getRefId() === null) {
                $ciObject->create()
                    ->createIdentifier();
                $categoryLogger->addRecord(CmdbRecord::objectCreated($ciObject->getRefId(), $ciObject->getClassId(), $title));
                $message = "External id {$extType}/{$extId} not found. Object with id {$ciObject->getRefId()} created.";
            } else {
                $message = "Object for external id {$extType}/{$extId} found: {$ciObject->getRefId()}";
            }

            $logger->info($message);

            foreach ($data as $categoryConstant => $categoryData) {
                /**
                 * @var $dao isys_cmdb_dao_category
                 */
                $descriptor = Descriptor::byConstant($categoryConstant);
                $sourceTable = $descriptor->getSourceTable();

                if (strpos($sourceTable, '_list') === false && strpos($sourceTable, '_2_') === false) {
                    $sourceTable .= '_list';
                }

                $dao = $descriptor->getDaoInstance();
                $dao->set_object_id($ciObject->getRefId());
                $dao->set_catgory_const($categoryConstant);
                $dao->set_source_table($sourceTable);

                $strategy = $categoryData['strategy'] ?? Category::STRATEGY_UPDATE;

                if (!$this->checkRights($dao, $categoryConstant, $strategy)) {
                    $logger->info("No permission to use strategy {$strategy} on category {$dao->getCategoryTitle()}.");
                    continue;
                }

                $logger->info("Preparing data for category '{$language->get($dao->getCategoryTitle())}' using strategy {$strategy}.");
                $category = new Category($dao, $strategyProvider->getStrategy($strategy));
                $formattedData[] = $category->prepare($ciObject, $categoryData['data']);
            }

            $this->setData($formattedData);

            $pushIdentifier = PushIdentifier::instance(isys_application::instance()->container->get('database'));
            $pushId = $pushIdentifier->getDataByReference(
                $ciObject->getRefId(), PushIdentifier::OBJECT
            )['isys_push_identifier__id'];
            $pushIdentifier->touch(
                $pushId,
                isys_application::instance()->container->get('session')->get_current_username()
            );
        } catch(\Exception $e) {
            throw new \Exception($e->getMessage());
        }

        return $this;
    }

    /**
     * @param isys_cmdb_dao_category $dao
     * @param string                 $categoryConstant
     * @param string                 $strategy
     *
     * @return bool
     */
    private function checkRights(isys_cmdb_dao_category $dao, string $categoryConstant, string $strategy)
    {
        $objectId = $dao->get_object_id();
        $categoryConstant = $dao->get_category_const();
        switch ($strategy) {
            case Category::STRATEGY_CREATE:
                $rightToCheck = [isys_auth::CREATE];
                break;
            case Category::STRATEGY_OVERWRITE:
                $rightToCheck = [isys_auth::DELETE, isys_auth::CREATE];
                break;
            case Category::STRATEGY_UPDATE:
            default:
                $rightToCheck = [isys_auth::EDIT];

                if ($dao->is_multivalued()) {
                    $rightToCheck[] = isys_auth::CREATE;
                }
                break;
        }

        try {
            foreach ($rightToCheck as $right) {
                isys_auth_cmdb::instance()
                    ->check_rights_obj_and_category($right, $objectId, $categoryConstant);
            }
            return true;
        } catch(Exception $e) {
            // @todo log missing rights to modify category in object
            return false;
        }
    }

    /**
     * @return bool
     * @throws \isys_exception_validation
     */
    public function sync()
    {
        /**
         * @var Category $category
         */
        foreach ($this->getData() as $category) {
            /**
             * @var Entry $entry
             */
            foreach ($category->getData() as $entry) {
                $dao = $category->getDao();
                Logger::instance()->info('Sync ' . $category->getDao()->get_category_const() . ' in object ' . $dao->get_object_id() . ' with import status ' . $entry->getImportStatus() . '.');
                $dao->sync(
                    $entry->getData(),
                    $dao->get_object_id(),
                    $entry->getImportStatus()
                );
            }
        }

        return true;
    }
}
