<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Kai Schubert-Altmann <kai.schubert-altmann@becon.de>
 * @license    http://www.i-doit.com/license
 */

use idoit\Component\Helper\Ip;

class isys_api_model_cmdb_ip_search extends isys_api_model_cmdb implements isys_api_model_interface, isys_api_model_option_interface
{
    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    private $layer3_obj_mapping = [
        'isys_obj__id'                           => 'id',
        'isys_obj__title'                        => 'title',
        'isys_obj__sysid'                        => 'sysid',
        'isys_obj__isys_cmdb_status__id'         => 'cmdb_status',
        'isys_cats_net_list__address_range_from' => 'range from',
        'isys_cats_net_list__address_range_to'   => 'range to',
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * @return string[]
     */
    public static function getAllowedOptions(): array
    {
        return [
            'read',
            'getNetworksByIp',
        ];
    }

    /**
     * @param array $p_params
     *
     * @return mixed
     * @throws isys_exception_api
     */
    public function read($p_params)
    {
        if (!isset($p_params["option"]) || !method_exists($this, $p_params["option"])) {
            throw new isys_exception_api("Required parameter option not set.");
        }

        // "option" is set to read by isys_api_model_cmdb in case the method is called via "cmdb.filter.read" just preventing an infinite loop here.
        if ($p_params["option"] !== 'read') {
            return call_user_func_array([$this, $p_params["option"]], [$p_params]);
        }

        return $this->getNetworksByIp($p_params);
    }

    /**
     * @param array $p_params
     *
     * @return null|void
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params
     *
     * @return null|void
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params
     *
     * @return null|void
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * Get all Layer3 Objects (except Global v4/v6, where the IP is in the address range of the Layer3 Network)
     *
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     * "ip" => "192.168.20.15" (valid IPv4)
     * OR
     * "ip" => "2001:0db8:85a3:0000:0000:8a2e:0370:7334" (valid IPv6)
     * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
     *
     * @param array $p_params
     *
     * @return array
     */
    public function getNetworksByIp($p_params)
    {
        $l_return = [];

        if (empty($p_params['ip'])) {
            throw new isys_exception_api("Parameter 'ip' must not be empty.");
        }

        $l_raw = false;
        if (isset($p_params['raw'])) {
            $l_raw = filter_var($p_params['raw'], FILTER_VALIDATE_BOOLEAN);
        }

        //Check if parameter 'ip' is valid IPv4
        if (Ip::validate_ip($p_params['ip'])) {
            $layer3_objects = $this->get_matching_nets_by_ipv4_address($p_params['ip'], $l_raw);

            //Check if parameter 'ip' is valid IPv6
        } elseif (Ip::validate_ipv6($p_params['ip'])) {
            $layer3_objects = $this->get_matching_nets_by_ipv6_address($p_params['ip'], $l_raw);
        } else {
            throw new isys_exception_api("Parameter 'ip' is not a valid IPv4 or IPv6.");

        }

        return $layer3_objects;
    }

    /**
     * Retrieve all layer 3 net objects for the given ipv4 address
     *
     * @param string $p_ip_address
     * @param bool   $p_raw
     *
     * @return array
     */
    private function get_matching_nets_by_ipv4_address(string $p_ip_address, bool $p_raw)
    {
        $l_objects = [];
        $l_sql = "SELECT *
            FROM isys_cats_net_list
            LEFT JOIN isys_obj ON isys_cats_net_list__isys_obj__id = isys_obj__id
            WHERE isys_obj__status = " . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . "
            AND isys_cats_net_list__status = " . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . "
            AND isys_cats_net_list__isys_obj__id != " . $this->m_dao->convert_sql_int(defined_or_default('C__OBJ__NET_GLOBAL_IPV4')) . "
            AND isys_cats_net_list__isys_net_type__id = " . $this->m_dao->convert_sql_int(defined_or_default('C__CATS_NET_TYPE__IPV4')) . "
            AND " . $this->m_dao->convert_sql_int(Ip::ip2long($p_ip_address)) . " BETWEEN isys_cats_net_list__address_range_from_long AND isys_cats_net_list__address_range_to_long";

        $l_result = $this->m_dao->retrieve($l_sql);

        while ($l_row = $l_result->get_row()) {
            if ($p_raw) {
                $l_objects[] = $l_row;
            } else {
                $l_objects[] = $this->format_by_mapping($this->layer3_obj_mapping, $l_row);
            }
        }

        return $l_objects;
    }

    /**
     * Retrieve all layer 3 net objects for the given ipv6 address
     *
     * @param string $p_ip_address
     * @param bool   $p_raw
     *
     * @return array
     */
    private function get_matching_nets_by_ipv6_address(string $p_ip_address, bool $p_raw)
    {

        $l_objects = [];
        $l_sql = "SELECT *
            FROM isys_cats_net_list
            LEFT JOIN isys_obj ON isys_cats_net_list__isys_obj__id = isys_obj__id
            WHERE isys_obj__status = " . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . "
            AND isys_cats_net_list__status = " . $this->m_dao->convert_sql_int(C__RECORD_STATUS__NORMAL) . "
            AND isys_cats_net_list__isys_obj__id != " . $this->m_dao->convert_sql_int(defined_or_default('C__OBJ__NET_GLOBAL_IPV6')) . "
            AND isys_cats_net_list__isys_net_type__id = " . $this->m_dao->convert_sql_int(defined_or_default('C__CATS_NET_TYPE__IPV6'));

        $l_result = $this->m_dao->retrieve($l_sql);

        while ($l_row = $l_result->get_row()) {
            if (Ip::is_ipv6_in_range($p_ip_address, $l_row['isys_cats_net_list__address_range_from'], $l_row['isys_cats_net_list__address_range_to'])) {
                if ($p_raw) {
                    $l_objects[] = $l_row;
                } else {
                    $l_objects[] = $this->format_by_mapping($this->layer3_obj_mapping, $l_row);
                }
            }
        }

        return $l_objects;
    }

    /**
     * Constructor
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}
