<?php

namespace idoit\Module\Api\Endpoint;

use Exception;

/**
 * Validate class for parameter related validation logic.
 */
class Validate
{
    /**
     * @param int $id
     *
     * @return bool
     * @throws Exception
     */
    public static function isId(int $id): bool
    {
        if ($id <= 0) {
            throw new Exception('The value may not be lower than 1.');
        }

        return true;
    }

    /**
     * @param array $ids
     *
     * @return bool
     * @throws Exception
     */
    public static function isArrayOfIds(array $ids): bool
    {
        $validIds = array_filter($ids, fn($id) => is_int($id) && $id > 0);

        if (count($validIds) !== count($ids)) {
            throw new Exception('At least one provided ID value was not an integer or lower than 1.');
        }

        return true;
    }

    /**
     * @param int|array $ids
     *
     * @return bool
     * @throws Exception
     */
    public static function isIdOrArrayOfIds(int|array $ids): bool
    {
        if (is_int($ids)) {
            self::isId($ids);
        } else {
            self::isArrayOfIds($ids);
        }

        return true;
    }

    /**
     * @param string $constant
     * @param string $prefix
     *
     * @return bool
     * @throws Exception
     */
    public static function isConstant(string $constant, string $prefix = 'C_', bool $mustBeDefined = false): bool
    {
        if (!str_starts_with($constant, $prefix) || $constant !== strtoupper($constant)) {
            throw new Exception("The value needs to begin with '{$prefix}' and be written in uppercase.");
        }

        if ($mustBeDefined && !defined($constant)) {
            throw new Exception("The category constant '{$constant}' is not defined in your i-doit installation.");
        }

        return true;
    }

    /**
     * @param array  $constants
     * @param string $prefix
     *
     * @return bool
     * @throws Exception
     */
    public static function isArrayOfConstants(array $constants, string $prefix = 'C_', bool $mustBeDefined = false): bool
    {
        $validConstants = array_filter($constants, fn($constant) => is_string($constant) && str_starts_with($constant, $prefix) && $constant === strtoupper($constant));

        if (count($validConstants) !== count($constants)) {
            throw new Exception("At least one provided constant was not an string, did not begin with '{$prefix}' or was not written in uppercase.");
        }

        if ($mustBeDefined) {
            foreach ($constants as $constant) {
                if (!defined($constant)) {
                    throw new Exception("The category constant '{$constant}' is not defined in your i-doit installation.");
                }
            }
        }

        return true;
    }

    /**
     * @param string|array $constants
     * @param string       $prefix
     *
     * @return bool
     * @throws Exception
     */
    public static function isConstantOrArrayOfConstants(string|array $constants, string $prefix = 'C_', bool $mustBeDefined = false): bool
    {
        if (is_string($constants)) {
            self::isConstant($constants, $prefix, $mustBeDefined);
        } else {
            self::isArrayOfConstants($constants, $prefix, $mustBeDefined);
        }

        return true;
    }
}
