<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Object;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Endpoint\Validate;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use isys_application;
use isys_auth;
use isys_module_cmdb;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object read endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object.read.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_READ',
            [
                new RequiredParameter('object', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__OBJECT_ID', [Validate::class, 'isId']),
                new OptionalParameter('categories', Parameter::TYPE_ARRAY, 'LC__ADDON__API__PARAMETER__CATEGORY_CONSTANTS', fn ($categories) => Validate::isArrayOfConstants($categories, 'C__CAT', true), [])
            ]
        );
    }

    public function request(Request $request): JsonRpcResponse
    {
        $objectId = $request->request->get('object');
        // @see API-598 Refactor usages of 'get' if the values contain arrays.
        $categories = $request->request->all('categories');

        isys_module_cmdb::getAuth()->check_rights_obj_and_category(isys_auth::VIEW, $objectId, 'C__CATG__GLOBAL');

        // Data retrieval.
        $objectResult = $this->daoCmdb->get_object_by_id($objectId);

        // Data formatting.
        if (count($objectResult) === 0) {
            throw new InternalErrorException('No object exists with ID ' . $objectId);
        }

        $objectRow = $objectResult->get_row();

        $routeGenerator = isys_application::instance()->container->get('route_generator');

        $responseData = [
            'id'                => (int)$objectRow['isys_obj__id'],
            'title'             => $this->language->get($objectRow['isys_obj__title']),
            'titleRaw'          => $objectRow['isys_obj__title'],
            'sysId'      => $objectRow['isys_obj__sysid'],
            'type'       => [
                'id'       => (int)$objectRow['isys_obj_type__id'],
                'constant' => $objectRow['isys_obj_type__const'],
                'title'    => $this->language->get($objectRow['isys_obj_type__title']),
                'titleRaw' => $objectRow['isys_obj_type__title'],
            ],
            // 'type-icon'         => $routeGenerator->generate('cmdb.object-type.icon', ['objectTypeId' => $objectRow['isys_obj_type__id']]),
            'status'            => $objectRow['isys_obj__status'],
            'cmdbStatus' => [
                'id'       => (int)$objectRow['isys_obj__isys_cmdb_status__id'],
                'title'    => $this->language->get($objectRow['isys_cmdb_status__title']),
                'titleRaw' => $objectRow['isys_cmdb_status__title'],
            ],
            'created'           => $objectRow['isys_obj__created'],
            'updated'           => $objectRow['isys_obj__updated'],
            // 'image'             => $routeGenerator->generate('cmdb.object.image', ['objectId' => $objectId])
        ];

        if (count($categories)) {
            $responseData['categories'] = [
                // @todo Read category data and apply it to the response data.
            ];
        }

        return new JsonRpcResponse($responseData);
    }
}
