<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use isys_component_dao_result;
use Symfony\Component\Filesystem\Filesystem;

/**
 * CMDB abstract object type group endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
abstract class AbstractObjectTypeGroupEndpoint extends AbstractCmdbEndpoint
{
    /**
     * @param int|array|null    $id
     * @param string|array|null $constant
     * @param bool|null         $visible
     *
     * @return isys_component_dao_result
     * @throws \isys_exception_database
     */
    protected function read(int|array|null $id = null, string|array|null $constant = null, bool|null $visible = null): isys_component_dao_result
    {
        $query = 'SELECT * FROM isys_obj_type_group WHERE TRUE';

        if ($id !== null) {
            if (is_int($id)) {
                $id = [$id];
            }

            if (is_array($id) && count($id) > 0) {
                $idList = implode(',', array_map('intval', $id));
                $query .= " AND isys_obj_type_group__id IN ({$idList})";
            }
        }

        if ($constant !== null) {
            if (is_string($constant)) {
                $constant = [$constant];
            }

            if (is_array($constant) && count($constant) > 0) {
                $constantList = implode(',', array_map([$this->daoCmdb, 'convert_sql_text'], $constant));
                $query .= " AND isys_obj_type_group__const IN ({$constantList})";
            }
        }

        if ($visible !== null) {
            $statusNormal = $this->daoCmdb->convert_sql_int(C__RECORD_STATUS__NORMAL);

            if ($visible) {
                $query .= " AND isys_obj_type_group__status = {$statusNormal}";
            } else {
                $query .= " AND isys_obj_type_group__status != {$statusNormal}";
            }
        }

        return $this->daoCmdb->retrieve($query);
    }

    /**
     * @param string $title
     *
     * @return string
     * @throws \isys_exception_database
     */
    private function getUniqueConstant(string $title): string
    {
        $proposedConstant = 'C__OBJTYPE_GROUP__SD_' . strtoupper(str_replace('-', '_', \isys_helper_upload::prepare_filename($title)));

        $checkedConstant = $proposedConstant;
        $count = 1;

        while ($this->read(null, $checkedConstant)->count() !== 0) {
            $checkedConstant = $proposedConstant . $count;
            $count++;
        }

        return $checkedConstant;
    }

    /**
     * @param int|null $id
     * @param array    $data
     *
     * @return bool
     * @throws \isys_exception_dao
     * @throws \isys_exception_database
     */
    protected function write(int|null $id = null, array $data = []): bool
    {
        $result = true;
        $change = [];

        if (isset($data['title']) && $data['title'] !== null) {
            $processedTitle = $this->daoCmdb->convert_sql_text($data['title']);
            $change[] = "isys_obj_type_group__title = {$processedTitle}";
        }

        // Every object type group needs a unique constant.
        if ($id === null) {
            $processedConstant = $this->daoCmdb->convert_sql_text($this->getUniqueConstant($data['title']));
            $change[] = "isys_obj_type_group__const = {$processedConstant}";
        }

        if (isset($data['sort']) && $data['sort'] !== null) {
            $processedSort = $this->daoCmdb->convert_sql_int($data['sort']);
            $change[] = "isys_obj_type_group__sort = {$processedSort}";
        }

        if (isset($data['visible']) && $data['visible'] !== null) {
            $processedVisibility = $this->daoCmdb->convert_sql_int($data['visible'] ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH);
            $change[] = "isys_obj_type_group__status = {$processedVisibility}";
        }

        if (count($change)) {
            $changes = implode(', ', $change);

            if ($id !== null) {
                $query = "UPDATE isys_obj_type_group
                    SET {$changes}
                    WHERE isys_obj_type_group__id = {$id}
                    LIMIT 1;";
            } else {
                $query = "INSERT INTO isys_obj_type_group
                    SET {$changes};";
            }

            $result = $this->daoCmdb->update($query) && $this->daoCmdb->apply_update();

            if ($result) {
                $this->clearCache();
            }
        }

        return $result;
    }

    /**
     * @param string $constant
     *
     * @return bool
     * @throws \isys_exception_dao
     */
    public function delete(string $constant): bool
    {
        if (!str_starts_with($constant, 'C__OBJTYPE_GROUP__SD_')) {
            throw new \Exception('Only custom object type groups can be deleted!');
        }

        $processedConstant = $this->daoCmdb->convert_sql_text($constant);

        $query = "DELETE FROM isys_obj_type_group
            WHERE isys_obj_type_group__const = {$processedConstant}
            LIMIT 1;";

        $result = $this->daoCmdb->update($query) && $this->daoCmdb->apply_update();

        if ($result) {
            $this->clearCache();
        }

        return $result;
    }

    /**
     * @return void
     * @throws \Exception
     */
    private function clearCache(): void
    {
        \isys_cache::keyvalue()->flush();

        (new Filesystem())->remove(glob(rtrim(isys_glob_get_temp_dir(), '/') . '/*'));

        \isys_application::instance()->container->get('signals')->emit('system.afterFlushSystemCache');
    }
}
