<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group delete endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Delete extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.object-type-group.delete.v2', 'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_DELETE', [
                new OptionalParameter('id', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__ID', [Validate::class, 'isId']),
                new OptionalParameter('constant', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CONSTANT',
                    function ($constant) { return Validate::isConstant($constant, 'C__OBJTYPE_GROUP__SD_'); }),
            ], [
                'With id'       => Helper::exampleRequest('cmdb.object-type-group.delete.v2', ['id' => 123]),
                'With constant' => Helper::exampleRequest('cmdb.object-type-group.delete.v2', ['constant' => 'C__OBJTYPE_GROUP__SD_CUSTOM_GROUP']),
            ], ['Basic response' => Helper::exampleResponse(['success' => true, 'messages' => "Object type constant 'Example group' (#123) successfully deleted."])]);
    }

    /**
     * @param Request $request
     *
     * @return JsonRpcResponse
     */
    public function request(Request $request): JsonRpcResponse
    {
        if ($request->request->get('id') === null && $request->request->get('constant') === null) {
            return new JsonRpcResponse(['success' => false, 'message' => 'You need to pass either "id" or "constant" to delete an object type group.']);
        }

        $objectTypeGroup = $this->read($request->request->get('id'), $request->request->get('constant'))
            ->get_row();

        if ($objectTypeGroup === null || empty($objectTypeGroup)) {
            return new JsonRpcResponse(['success' => false, 'message' => 'Object type group could not be found.']);
        }

        try {
            $id = $objectTypeGroup['isys_obj_type_group__id'];
            $title = $this->language->get($objectTypeGroup['isys_obj_type_group__title']);

            // In order to check for 'custom' object type groups, we work with the constant.
            $this->delete($objectTypeGroup['isys_obj_type_group__const']);

            return new JsonRpcResponse(['success' => true, 'messages' => "Object type constant '{$title}' (#{$id}) successfully deleted."]);
        } catch (\Throwable $e) {
            return new JsonRpcResponse(['success' => false, 'messages' => $e->getMessage()]);
        }
    }
}
