<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\ResponseModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_cmdb_dao_category_g_network_port;
use isys_request;

/**
 * PortProcessor
 *
 * @package    idoit\Module\Api\Model\Category
 */
class PortProcessor extends AbstractCategoryProcessor implements SyncModifier, ResponseModifier
{
    /**
     * Get cable property name
     *
     * @return string
     */
    protected function getCablePropertyName()
    {
        return 'cable';
    }

    /**
     * Get connector property name
     *
     * @return string
     */
    protected function getConnectorPropertyName() {
        return 'assigned_connector';
    }

    /**
     * @return array
     */
    public function getAffectedPropertiesBySync()
    {
        return [
            'cable',
            'assigned_connector',
            'layer2_assignment',
            'default_vlan',
        ];
    }

    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     */
    public function modifySyncData(array $syncData)
    {
        // Get request
        $request = $this->getRequest();

        $entryId = $request['data']['category_id'] ?? $request['data']['id'] ?? null;

        // Skip create requests
        if ($request['option'] == 'create') {
            return $syncData;
        }

        // Property name of cable object
        $cablePropertyName = $this->getCablePropertyName();

        /**
         * The following will convert 'cable' field which will present
         * a cable connection value into a valid cable object id.
         *
         * @see API-7
         */

        // Check whether cable id was setted explicitly or is already persisted
        if (empty($request['data'][$cablePropertyName])) {
            if (!empty($syncData['properties'][$cablePropertyName][C__DATA__VALUE])) {
                try {
                    // Create cable connection dao
                    $cableConnectionDao = new \isys_cmdb_dao_cable_connection(\isys_application::instance()->container->get('database'));

                    // Get cable object id by cable connection id
                    $syncData['properties'][$cablePropertyName][C__DATA__VALUE] = $cableConnectionDao->get_cable_object_id_by_connection_id($syncData['properties'][$cablePropertyName][C__DATA__VALUE]);
                } catch (\Exception $e) {
                    // No need to interrupt api call
                }
            }
        }

        /**
         * This routine will translate "assigned_connector" which will be simply the category entry id if it is unspecified
         * to the id of the assigned connector. Thereby we can prevent losing of the assigned connector during category update
         *
         * @see API-30
         */

        // Get assigned connector property name
        $connectorPropertyName = $this->getConnectorPropertyName();

        // Check whether it is empty
        if (empty($request['data'][$connectorPropertyName]) && !empty($syncData['properties'][$connectorPropertyName][C__DATA__VALUE])) {
            try {
                // Create cable connection dao
                $cableConnectionDao = new \isys_cmdb_dao_cable_connection(\isys_application::instance()->container->get('database'));

                // Get assigned connector
                $assignedConnectorId = $cableConnectionDao->get_assigned_connector($syncData['properties'][$connectorPropertyName][C__DATA__VALUE])->get_row_value('isys_catg_connector_list__id');

                // Check whether connector is assigned
                if (!empty($assignedConnectorId)) {
                    // Set assigned connector to prevent deletion while sync routine
                    $syncData['properties'][$connectorPropertyName][C__DATA__VALUE] = $assignedConnectorId;
                }
            } catch (\Exception $e) {
                // No need to interrupt api call
            }
        }

        $dao = $this->getDao();

        if ($dao instanceof isys_cmdb_dao_category_g_network_port) {
            // @see API-353 Fix the layer2 assignment
            // @see API-380 Changed from '$this instanceof PortProcessor' because the ConnectorProcessor extends this class.
            if (!isset($request['data']['layer2_assignment']) && !isset($syncData['properties']['layer2_assignment'])) {
                $requestInstance = isys_request::factory()->set_row(['isys_catg_port_list__id' => $entryId]);
                $syncData['properties']['layer2_assignment'][C__DATA__VALUE] = $dao->callback_property_layer2_assignment($requestInstance);
            }

            // @see API-353 Fix the default vlan assignment
            // @see API-380 Changed from '$this instanceof PortProcessor' because the ConnectorProcessor extends this class.
            if (!isset($request['data']['default_vlan'])) {
                $vlanResult = $dao->get_attached_layer2_net($entryId);

                while ($row = $vlanResult->get_row()) {
                    if ($row['default_vlan'] > 0) {
                        $syncData['properties']['default_vlan'][C__DATA__VALUE] = $row['object_id'];
                    }
                }
            }

            // @see API-359 Detach the assignments, if the data is explicitly set empty.
            if (array_key_exists('addresses', $request['data']) && $syncData['data_id'] > 0) {
                // Empty the attachments, if the 'addresses' contain NULL or an empty array.
                if ($request['data']['addresses'] === null || (is_array($request['data']['addresses']) && count($request['data']['addresses']) === 0)) {
                    $dao->clear_ip_attachments($syncData['data_id']);
                }
            }
        }

        return $syncData;
    }

    /**
     * @param array $response
     *
     * @return array
     */
    public function modifyResponse(array $response)
    {
        $request = $this->getRequest();

        if ($request['option'] === 'read') {
            $response = array_map(function ($entry) {
                // @see API-552 Remove the 'connector' because it contains misleading data.
                unset($entry['connector']);

                // @see API-384 Flatten the 'cable' array.
                if (is_array($entry['cable'])) {
                    if (count($entry['cable']) === 0) {
                        $entry['cable'] = null;
                    } else {
                        $entry['cable'] = current($entry['cable']);
                    }
                }

                return $entry;
            }, $response);
        }

        return $response;
    }
}
