<?php

use idoit\Module\Report\Report;
use idoit\Module\Report\Worker\ArrayWorker;

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_reports extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_report__id'           => 'id',
        'category_title'            => 'category',
        'isys_report__title'        => 'title',
        'isys_report__description'  => 'description',
        'isys_report__datetime'     => 'created',
        'isys_report__last_editied' => 'modified',
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read'   => [
            'id' => [
                'type'        => 'int',
                'description' => 'Report ID',
                'reference'   => 'isys_report__id',
                'optional'    => true
            ]
        ],
        'create' => [],
        'update' => [],
        'delete' => [],
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * Fetches information about an object.
     * [
     *    integer  $p_params['id']  The ID of the report
     * ]
     *
     * @param array $p_params
     *
     * @return array Returns an empty array when an error occures.
     * @throws isys_exception_api
     */
    public function read($p_params)
    {
        $l_return = [];

        $l_dao = isys_report_dao::instance(isys_application::instance()->container->get('database_system'));

        $offset = 0;
        $limit = null;

        // @see API-319 Implement 'offset' and 'limit'.
        if (isset($p_params['offset']) && is_numeric($p_params['offset'])) {
            $offset = (int)$p_params['offset'];
        }

        if (isset($p_params['limit']) && is_numeric($p_params['limit'])) {
            $limit = (int)$p_params['limit'];
        }

        if (isset($p_params['id'])) {
            $this->m_log->info('Retrieving report with id ' . $p_params['id']);

            if ($this->useAuth) {
                isys_auth_report::instance()
                    ->custom_report(isys_auth::VIEW, $p_params['id']);
            }

            $report = $l_dao->get_reports(null, [$p_params['id']], null, true, false);

            if (isset($report[0]) && !empty($report[0]['isys_report__query'])) {
                // @see API-551 Only rewrite the query, if we have querybuilder data.
                if (!empty($report[0]['isys_report__querybuilder_data']) && isys_format_json::is_json_array($report[0]['isys_report__querybuilder_data'])) {
                    $report[0]['isys_report__query'] = isys_cmdb_dao_category_property::instance($l_dao->get_database_component())
                        ->prepareEnvironmentForReportById($p_params['id'])
                        ->create_property_query_for_report() . '';
                }

                if (!$l_dao->hasQueryOnlyInternalPlaceholder($report[0]['isys_report__query'])) {
                    throw new isys_exception_api(sprintf('Report with id %s contains non internal placeholders, therefore cannot be executed by the api', $p_params['id']));
                }

                $query = $l_dao->replacePlaceHolders($report[0]['isys_report__query']);

                // @see API-370 Process the report with the 'ArrayWorker'.
                $worker = new ArrayWorker();
                $showHtml = (bool)$report[0]['isys_report__show_html'];
                // @see API-373 Check the 'compress multivalue results' option and set it later.
                $compressMultivalue = (bool)$report[0]['isys_report__compressed_multivalue_results'];

                (new Report($l_dao, $query))
                    ->enableHtmlContext($showHtml)
                    ->setCompressedMultivalueResults($compressMultivalue)
                    ->setWorker($worker)
                    ->execute();

                // @see API-428 Pass an empty array, if '$worker->export()' results in NULL.
                $result = array_slice($worker->export() ?? [], $offset, $limit);

                $removeScriptCallback = fn ($string) => $string;

                // @see API-370 Even though we want to see HTML we should remove 'script' tags.
                if ($showHtml) {
                    // Took this line from 'Report->execute()' when HTML context is deactivated.
                    $removeScriptCallback = fn ($string) => preg_replace("(<script[^>]*>([\\S\\s]*?)<\/script>)", '', $string);
                }

                foreach ($result as &$row) {
                    foreach ($row as &$column) {
                        // @see API-458 Replace '&nbsp;' with actual whitespaces and then trim the content.
                        $column = trim(str_replace('&nbsp;', ' ', $removeScriptCallback($column)));
                    }
                }

                return $result;
            }

            throw new isys_exception_api(sprintf('Report with id %s does not exist', $p_params['id']));
        }

        $reports = $l_dao->get_reports(null, null, null, true, false);

        foreach ($reports as $report) {
            if ($this->useAuth) {
                try {
                    isys_auth_report::instance()->custom_report(isys_auth::VIEW, $report['isys_report__id']);
                } catch (isys_exception_auth $e) {
                    $this->m_log->error($e->getMessage());

                    continue;
                }
            }

            // Skip reports with placeholders
            if (!$l_dao->hasQueryOnlyInternalPlaceholder($report['isys_report__query'])) {
                continue;
            }

            $l_return[] = $this->format_by_mapping($this->m_mapping, $report);
        }

        return $l_return;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * isys_api_model_cmdb_reports constructor.
     *
     * @param isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        parent::__construct();
    }
}
