<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\External;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Endpoint\Validate;
use idoit\Module\Api\PushIdentifier\PushIdentifier;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB endpoint external link
 *
 * @see       API-609
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Link extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.external.link.v2', 'LC__ADDON__API__ENDPOINT__CMDB_EXTERNAL_LINK', [
            new RequiredParameter('extType', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_EXT_TYPE', fn ($type) => trim($type) !== ''),
            new RequiredParameter('extId', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_EXT_ID', fn ($id) => trim($id) !== ''),
            new RequiredParameter('objectId', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__OBJECT_ID', [Validate::class, 'isId']),
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        try {
            $objectId = $request->get('objectId');
            $externalType = $request->get('extType');
            $externalId = $request->get('extId');

            // Check if given object ID exists.
            $objectData = $this->daoCmdb->get_object($objectId)->get_row();

            if ($objectData === null) {
                throw new \Exception("Object with ID {$objectId} does not exist.");
            }

            $pushIdentifier = PushIdentifier::instance($this->database);

            if ($pushIdentifier->referenceExists($objectId)) {
                $existingIdentifier = $pushIdentifier->getDataByReference($objectId, PushIdentifier::OBJECT);

                if ($existingIdentifier['isys_push_identifier__external_type'] === $externalType && $existingIdentifier['isys_push_identifier__external_id'] === $externalId) {
                    return new JsonRpcResponse(['success' => true, 'message' => 'External identifier is already linked to given object.']);
                }

                throw new \Exception("Object with ID {$objectId} is already known by external identifier '{$existingIdentifier['isys_push_identifier__external_type']}/{$existingIdentifier['isys_push_identifier__external_id']}'.");
            }

            $externalIdentifiers = $pushIdentifier->findReferences($externalType, $externalId);

            if (count($externalIdentifiers)) {
                throw new \Exception("External identifier '{$externalType}/{$externalId}' already exists.'");
            }

            $objectSuccess = $pushIdentifier->createReference(
                $externalType,
                $externalId,
                PushIdentifier::OBJECT,
                $objectId,
                \isys_application::instance()->container->get('session')->get_current_username()
            );

            if (!$objectSuccess) {
                throw new \Exception("Unable to create object reference entry.");
            }

            $categorySuccess = $pushIdentifier->createReference(
                "{$externalType}/{$externalId}/C__CATG__GLOBAL",
                $objectId,
                "C__CATG__GLOBAL",
                $objectId,
                \isys_application::instance()->container->get('session')->get_current_username()
            );

            if (!$categorySuccess) {
                throw new \Exception("Unable to create category reference entry.");
            }

            return new JsonRpcResponse(['success' => true]);
        } catch (\Exception $e) {
            return new JsonRpcResponse(['messages' => $e->getMessage(), 'success' => false], 400);
        }
    }
}
