<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\External;

use Exception;
use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Endpoint\Validate;
use idoit\Module\Api\Model\External\Config;
use idoit\Module\Api\Model\External\Logger;
use idoit\Module\Api\Model\External\Push as PushModel;
use idoit\Module\Cmdb\Component\CategoryLogger\CmdbLogger;
use Monolog\LogRecord;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB endpoint external push
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Push extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.external.push.v2', 'LC__ADDON__API__ENDPOINT__CMDB_EXTERNAL_PUSH', [
            new RequiredParameter('extType', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_EXT_TYPE', fn ($type) => trim($type) !== ''),
            new RequiredParameter('extId', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_EXT_ID', fn ($id) => trim($id) !== ''),
            new RequiredParameter('title', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_OBJECT_TITLE', fn ($title) => trim($title) !== ''),
            new RequiredParameter('class', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__EXTERNAL_OBJECT_TYPE_TITLE', fn ($class) => Validate::isConstant($class, 'C__OBJTYPE__', true)),
            new RequiredParameter('data', Parameter::TYPE_ARRAY, 'LC__ADDON__API__PARAMETER__EXTERNAL_CATEGORY_DATA', function ($data) {
                if (empty($data)) {
                    throw new Exception('The passed data needs to contain valid category data: {"C__CATG__MODEL" => { "strategy" => "...", "data" => [ ... ] } }.');
                }

                foreach ($data as $categoryConstant => $categoryData) {
                    if (!str_starts_with($categoryConstant, 'C__CAT') || !defined($categoryConstant)) {
                        throw new Exception('The passed data needs to contain at least one category: \'{"C__CATG__MODEL" => { ... } }\'.');
                    }

                    if (!isset($categoryData['strategy'])) {
                        throw new Exception('The passed data needs to contain a strategy for every category: {"C__CATG__MODEL" => { "strategy" => "..." } }.');
                    }

                    if (empty($categoryData['data'])) {
                        throw new Exception('The passed data needs to contain data for every category: {"C__CATG__MODEL" => { "data" => [ ... ] } }.');
                    }
                }

                return true;
            }),
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        try {
            $config = new Config($request->get('extType'), $request->get('extId'), $request->get('title'), $request->get('class'), $request->get('data'));
            $success = (new PushModel($config))->prepare()
                ->sync();

            CmdbLogger::instance()
                ->writeCmdbLog(defined_or_default('C__LOGBOOK_SOURCE__API_PUSH', 0));

            return new JsonRpcResponse([
                'messages' => array_map(
                    function ($record) {
                        // @see API-596 Monolog 3 compatibility.
                        if ($record instanceof LogRecord) {
                            return $record->toArray();
                        }

                        return $record;
                    },
                    Logger::instance()->getRecords()
                ),
                'success'  => true
            ]);
        } catch (\Exception $e) {
            return new JsonRpcResponse(['messages' => $e->getMessage(), 'success' => false], 400);
        }
    }
}
