<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group read endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type-group.read.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_READ',
            [
                new OptionalParameter('id', [Parameter::TYPE_INTEGER, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__ID_OR_ARRAY', [Validate::class, 'isIdOrArrayOfIds']),
                new OptionalParameter('constant', [Parameter::TYPE_STRING, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__CONSTANT_OR_ARRAY', [Validate::class, 'isConstantOrArrayOfConstants']),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_VISIBLE')
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type-group.read.v2'),
                'With ids' => Helper::exampleRequest('cmdb.object-type-group.read.v2', ['id' => 1]),
                'With constants' => Helper::exampleRequest('cmdb.object-type-group.read.v2', ['constant' => 'C__OBJTYPE_GROUP__SOFTWARE']),
            ],
            ['Basic response' => Helper::exampleResponse([
                [
                    'id'        => 1,
                    'constant'  => 'C__OBJTYPE_GROUP__SOFTWARE',
                    'title'     => 'Software',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__SOFTWARE',
                    'sort'      => 1,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                [
                    'id'        => 2,
                    'constant'  => 'C__OBJTYPE_GROUP__INFRASTRUCTURE',
                    'title'     => 'Infrastructure',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__INFRASTRUCTURE',
                    'sort'      => 2,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                [
                    'id'        => 3,
                    'constant'  => 'C__OBJTYPE_GROUP__OTHER',
                    'title'     => 'Other',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__OTHER',
                    'sort'      => 3,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                ['...']
            ])
            ]
        );
    }

    /**
     * @param Request $request
     *
     * @return JsonRpcResponse
     */
    public function request(Request $request): JsonRpcResponse
    {
        $objectTypeGroups = [];

        // @see API-598 Refactor usages of 'get' if the values can contain arrays.
        $parameters = $request->request->all();

        $objectTypeGroupResult = $this->read(
            $parameters['id'] ?? null,
            $parameters['constant'] ?? null,
            $request->request->get('visible')
        );

        while ($row = $objectTypeGroupResult->get_row()) {
            // We hide the unused fields 'isys_obj_type_group__description' and 'isys_obj_type_group__property'.
            $objectTypeGroups[] = [
                'id'          => (int)$row['isys_obj_type_group__id'],
                'constant'    => $row['isys_obj_type_group__const'],
                'title'       => $this->language->get($row['isys_obj_type_group__title']),
                'titleRaw'    => $row['isys_obj_type_group__title'],
                'sort'        => (int)$row['isys_obj_type_group__sort'],
                'status'      => (int)$row['isys_obj_type_group__status'],
                'visible'     => $row['isys_obj_type_group__status'] == C__RECORD_STATUS__NORMAL,
                'deletable'   => str_starts_with($row['isys_obj_type_group__const'], 'C__OBJTYPE_GROUP__SD_')
            ];
        }

        return new JsonRpcResponse($objectTypeGroups);
    }
}
