<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use isys_auth;
use isys_module_system;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group update endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Update extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type-group.update.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_UPDATE',
            [
                new RequiredParameter('id', [Parameter::TYPE_INTEGER, Parameter::TYPE_STRING], 'LC__ADDON__API__PARAMETER__OBJECT_TYPE_GROUP_CONSTANT_OR_ID', fn ($identifier) => is_int($identifier) ? Validate::isId($identifier) : Validate::isConstant($identifier, 'C__OBJTYPE_GROUP__', true)),
                new OptionalParameter('title', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_TITLE', fn ($title) => trim($title) !== ''),
                new OptionalParameter('sort', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_SORT'),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_VISIBLE')
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'title' => 'New object type group name']),
                'Change visibility' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'visible' => false]),
                'Change order' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'sort' => 99]),
            ],
            ['Basic response' => Helper::exampleResponse(['success' => true])]
        );
    }

    /**
     * @param Request $request
     *
     * @return JsonRpcResponse
     * @throws InternalErrorException
     * @throws \isys_exception_dao
     */
    public function request(Request $request): JsonRpcResponse
    {
        isys_module_system::getAuth()->check(isys_auth::EXECUTE, 'globalsettings/qcw');

        $identifier = $request->request->get('id');

        $objectTypeId = $this
            ->read(is_int($identifier) ? $identifier : null, is_string($identifier) && defined($identifier) ? $identifier : null)
            ->get_row_value('isys_obj_type_group__id');

        if ($objectTypeId === null || empty($objectTypeId)) {
            throw new InternalErrorException('No object type group exists with ID / constant ' . $identifier);
        }

        $data = [
            'title'   => $request->request->get('title'),
            'sort'    => $request->request->get('sort'),
            'visible' => $request->request->get('visible'),
        ];

        return new JsonRpcResponse(['success' => $this->write((int)$objectTypeId, $data)]);
    }
}
