<?php

namespace idoit\Module\Api\Model\External\Handler;

use idoit\Component\Property\Property;
use idoit\Module\Cmdb\Component\CategoryLogger\CmdbLogger;
use idoit\Module\Cmdb\Component\CategoryLogger\CmdbRecord;
use idoit\Module\Api\Model\External\Logger;
use idoit\Module\Api\Model\External\Push\Entry;
use idoit\Module\Api\PushIdentifier\PushIdentifier;
use idoit\Module\Cmdb\Component\SyncMerger\Config;
use idoit\Module\Cmdb\Component\SyncMerger\Merger;
use idoit\Module\Cmdb\Component\SyncNormalizer\Config as NormalizerConfig;
use idoit\Module\Cmdb\Component\SyncNormalizer\Normalizer;
use isys_application;
use isys_auth;
use isys_auth_cmdb;
use isys_cmdb_dao_category;
use isys_cmdb_dao_category_g_contact;
use isys_cmdb_dao_category_g_custom_fields;
use isys_format_json;
use isys_import_handler_cmdb;

class Category
{
    /**
     * @var CiObject
     */
    protected CiObject $ciObject;

    /**
     * @var isys_cmdb_dao_category
     */
    protected isys_cmdb_dao_category $categoryDao;

    /**
     * @var string
     */
    protected string $extType;

    /**
     * @var string
     */
    protected string $refType;

    /**
     * @param CiObject               $ciObject
     * @param isys_cmdb_dao_category $categoryDao
     * @param string                 $extType
     * @param string|null            $refType
     */
    public function __construct(CiObject $ciObject, isys_cmdb_dao_category $categoryDao, string $extType, string $refType)
    {
        $this->ciObject = $ciObject;
        $this->categoryDao = $categoryDao;
        $this->extType = $extType;
        $this->refType = $refType;
    }

    /**
     * @return CiObject
     */
    public function getCiObject(): CiObject
    {
        return $this->ciObject;
    }

    /**
     * @return isys_cmdb_dao_category
     */
    public function getCategoryDao(): isys_cmdb_dao_category
    {
        return $this->categoryDao;
    }

    /**
     * @return int|null
     */
    public function findSingleValueEntryId(): ?int
    {
        if (!$this->categoryDao->is_multivalued()
            && ($row = $this->categoryDao->get_data(null, $this->ciObject->getRefId())->get_row())
        ) {
            return (int) $row[$this->categoryDao->get_source_table() . '__id'];
        }
        return null;
    }

    /**
     * @param string $externalId
     *
     * @return int|null
     * @throws \Exception
     */
    public function findRefIdByExternalId(string $externalId): ?int
    {
        $externalType = $this->extType;
        $identifierData = (PushIdentifier::instance(isys_application::instance()->container->get('database')))
            ->findReferences($externalType, $externalId);

        if (!empty($identifierData)) {
            return (int) current($identifierData)['referenceId'];
        }

        return null;
    }

    /**
     * @return array
     * @throws \Exception
     */
    public function find(): array
    {
        return (PushIdentifier::instance(isys_application::instance()->container->get('database')))
            ->findExternalIds($this->extType);
    }

    /**
     * @return int|null
     * @throws \isys_exception_auth
     * @throws \isys_exception_general
     */
    public function create(): ?int
    {
        if (isys_auth_cmdb::instance()->category(isys_auth::CREATE, $this->categoryDao->get_category_const())) {
            $fakeEntry = [
                Config::CONFIG_DATA_ID => null,
                Config::CONFIG_PROPERTIES => []
            ];

            $currentData = Merger::instance(Config::instance($this->categoryDao, $this->getCiObject()->getRefId(), $fakeEntry))->getDataForSync();
            $this->modifySyncData($currentData);

            return $this->categoryDao->sync(
                $currentData,
                $this->getCiObject()->getRefId(),
                isys_import_handler_cmdb::C__CREATE
            );
        }
        return null;
    }

    /**
     * @param array $syncData
     *
     * @return void
     */
    private function modifySyncData(array &$syncData){
        if ($this->categoryDao instanceof isys_cmdb_dao_category_g_contact) {
            $syncData[Config::CONFIG_PROPERTIES]['contact']['value'] = C__OBJ__ROOT_LOCATION;
            return;
        }

        if ($this->categoryDao->get_connected_object_id_field() === 'isys_connection__isys_obj__id') {
            $properties = $this->categoryDao->get_properties();
            foreach ($properties as $key => $property) {
                if (!$property instanceof Property) {
                    $property = Property::createInstanceFromArray($property);
                }

                if (strpos($property->getData()->getField(), 'isys_connection') !== false
                    && isset($syncData[Config::CONFIG_PROPERTIES][$key]['value'])
                ) {
                    $syncData[Config::CONFIG_PROPERTIES][$key]['value'] = C__OBJ__ROOT_LOCATION;
                    return;
                }
            }
        }
    }

    /**
     * @return $this
     * @throws \Exception
     */
    public function createIdentifier(string $externalId, int $internalId): Category
    {
        (PushIdentifier::instance(isys_application::instance()->container->get('database')))
            ->createReference(
                $this->extType,
                $externalId,
                $this->refType,
                $internalId,
                isys_application::instance()->container->get('session')->get_current_username()
            );
        return $this;
    }

    /**
     * @param array $externalIds
     *
     * @return bool
     * @throws \Exception
     */
    public function deleteByExternalIds(array $externalIds): bool
    {
        if (empty($externalIds)) {
            return false;
        }

        return (PushIdentifier::instance(isys_application::instance()->container->get('database')))
            ->deleteByExternalData($this->extType, $externalIds);
    }

    /**
     * @param array $ids
     *
     * @return bool
     */
    public function deleteCategoryEntriesByIds(array $ids): bool
    {
        if (empty($ids)) {
            return false;
        }

        return $this->categoryDao->rank_records($ids, C__CMDB__RANK__DIRECTION_DELETE, $this->categoryDao->get_source_table(), null, true);
    }

    /**
     * @param int $internalId
     * @param     $entry
     *
     * @return array
     */
    private function normalizeData(int $internalId, $entry)
    {
        $config = new NormalizerConfig($this->categoryDao, $entry);
        $normalizer = new Normalizer($config);
        return $normalizer->normalizeData($this->ciObject->getRefId(), $internalId);
    }

    /**
     * @param int $internalId
     * @param     $entry
     *
     * @return Entry
     */
    public function buildEntry(int $internalId, $entry, $created = false)
    {
        $importStatus = isys_import_handler_cmdb::C__UPDATE;
        $dao = $this->categoryDao;
        Logger::instance()->debug('Original: ' . isys_format_json::encode($entry));
        $entry = $this->normalizeData($internalId, $entry);
        Logger::instance()->debug('Normalized: ' . isys_format_json::encode($entry));
        $cmdbLogger = CmdbLogger::instance();

        if ($created) {
            // If entry has been created retrieve all default values beforehand
            $emptyData = Merger::instance(Config::instance($dao, $this->ciObject->getRefId(), [
                Config::CONFIG_DATA_ID    => null,
                Config::CONFIG_PROPERTIES => []
            ]))
                ->getDataForSync();

            $mergedData = array_replace_recursive($emptyData, [
                Config::CONFIG_DATA_ID    => $internalId,
                Config::CONFIG_PROPERTIES => array_map(function ($prop) {
                    return [C__DATA__VALUE => $prop];
                }, $entry)
            ]);

            $cmdbLogger->addRecord(CmdbRecord::categoryEntryCreated(
                $this->ciObject->getRefId(),
                $this->ciObject->getClassId(),
                $dao->getCategoryTitle(),
                $cmdbLogger->getLogbookDao()->prepare_changes($dao, null, $mergedData)
            ));

        } else {
            $dataToBeSynced = [
                Config::CONFIG_DATA_ID    => $internalId,
                Config::CONFIG_PROPERTIES => array_map(function ($prop) {
                    return [C__DATA__VALUE => $prop];
                }, $entry)
            ];

            $mergedData = Merger::instance(Config::instance($dao, $this->ciObject->getRefId(), $dataToBeSynced))->getDataForSync();
            $changes = $cmdbLogger->getLogbookDao()->prepare_changes($dao, $dao->get_data($internalId)->get_row(), $mergedData);

            if (count($changes)) {
                $cmdbLogger->addRecord(CmdbRecord::categoryEntryUpdated(
                    $this->ciObject->getRefId(),
                    $this->ciObject->getClassId(),
                    $dao->getCategoryTitle(),
                    $changes
                ));
            }
        }

        $stringifiedData = isys_format_json::encode($mergedData);
        Logger::instance()->debug("Final: ". isys_format_json::encode($mergedData));

        return new Entry($importStatus, $mergedData);
    }
}
