<?php

/**
 * JsonRPC
 *
 * @package    i-doit
 * @subpackage API
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

// Include ApiEnvironmentShell.
use idoit\Module\Api\ApiEnvironmentShell;
use idoit\Module\Api\Controller\JsonRpcController;
use idoit\Module\Api\Exception\JsonRpc\AbstractJsonRpcException;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\RequestException;
use idoit\Module\Api\JsonRpcOutput;
use idoit\Module\Api\Request\RequestStore;
use idoit\Module\Api\Response\ResponseStore;
use idoit\Module\Api\Session\SessionIdSession;

include __DIR__ . '/src/ApiEnvironmentShell.php';

/**
 * Display an json rpc error.
 *
 * @param string $message
 * @param int    $errorCode
 * @param mixed  $data
 */
function jsonRpcErrorOutput($message, $errorCode = null, $data = null)
{
    header('Content-Type: application/json');

    echo json_encode([
        'id'      => null,
        'jsonrpc' => '2.0',
        'error'   => [
            'code'    => $errorCode ?: InternalErrorException::CODE,
            'message' => $message,
            'data'    => $data
        ]
    ]);
}

// Create environment shell.
$apiEnvironmentShell = new ApiEnvironmentShell($idoitRootDirectory);

// Setup boot error handler.
$apiEnvironmentShell->injectBootErrorHandler(function (\Exception $e) {
    /**
     * Boot sequence exited with errors
     *
     * There is no guarantee that autoload and other
     * fundamental application services were loaded correctly.
     *
     * Therefore, do not access them to prevent further errors.
     */
    jsonRpcErrorOutput('i-doit System Error: ' . $e->getMessage());
});

// Setup error handler.
$apiEnvironmentShell->injectErrorHandler(function (\Exception $e) {
    if ($e instanceof AbstractJsonRpcException) {
        jsonRpcErrorOutput($e->getErrorTopic() . ' : ' . $e->getMessage(), $e->getErrorCode(), $e->getData());

        return;
    }

    jsonRpcErrorOutput('i-doit System Error: ' . $e->getMessage());
});

// Setup executable.
$apiEnvironmentShell->injectExecutable(function (ApiEnvironmentShell $apiShell) {
    // Check whether api init file exists.
    if (!file_exists($apiShell->getIdoitRootDirectoryPath() . '/src/classes/modules/api/init.php')) {
        throw new Exception('Api Module is not available.');
    }

    require_once $apiShell->getIdoitRootDirectoryPath() . '/src/classes/modules/api/init.php';

    // Check whether JsonRpcController exists.
    if (!class_exists(JsonRpcController::class)) {
        throw new Exception('Error: i-doit is unavailable.');
    }

    // Validate request parameters.
    if (JsonRpcController::validateHttpRequest()) {
        // @todo At some point we should remove this. Originally it should be removed in 1.13
        $categoryConstantMapping = [
            'C__CMDB__SUBCAT__NETWORK_PORT'                      => 'C__CATG__NETWORK_PORT',
            'C__CMDB__SUBCAT__NETWORK_INTERFACE_P'               => 'C__CATG__NETWORK_INTERFACE',
            'C__CMDB__SUBCAT__NETWORK_INTERFACE_L'               => 'C__CATG__NETWORK_LOG_PORT',
            'C__CMDB__SUBCAT__NETWORK_PORT_OVERVIEW'             => 'C__CATG__NETWORK_PORT_OVERVIEW',
            'C__CMDB__SUBCAT__STORAGE__DEVICE'                   => 'C__CATG__STORAGE_DEVICE',
            'C__CMDB__SUBCAT__LICENCE_LIST'                      => 'C__CATS__LICENCE_LIST',
            'C__CMDB__SUBCAT__LICENCE_OVERVIEW'                  => 'C__CATS__LICENCE_OVERVIEW',
            'C__CMDB__SUBCAT__EMERGENCY_PLAN_LINKED_OBJECT_LIST' => 'C__CATS__EMERGENCY_PLAN_LINKED_OBJECTS',
            'C__CMDB__SUBCAT__EMERGENCY_PLAN'                    => 'C__CATS__EMERGENCY_PLAN_ATTRIBUTE',
            'C__CMDB__SUBCAT__WS_NET_TYPE'                       => 'C__CATS__WS_NET_TYPE',
            'C__CMDB__SUBCAT__WS_ASSIGNMENT'                     => 'C__CATS__WS_ASSIGNMENT',
            'C__CMDB__SUBCAT__FILE_OBJECTS'                      => 'C__CATS__FILE_OBJECTS',
            'C__CMDB__SUBCAT__FILE_VERSIONS'                     => 'C__CATS__FILE_VERSIONS',
            'C__CMDB__SUBCAT__FILE_ACTUAL'                       => 'C__CATS__FILE_ACTUAL'
        ];

        $rawJsonRequest = strtr(file_get_contents('php://input'), $categoryConstantMapping);

        try {
            // Decode json body.
            $decodedRequest = isys_format_json::decode($rawJsonRequest);
        } catch (\Exception $e) {
            throw new RequestException("Provided request is not a valid JSON, got: {$rawJsonRequest} ({$e->getMessage()})");
        }

        // Check whether decoded request is an array.
        if (!is_array($decodedRequest)) {
            $decodedType = gettype($decodedRequest);
            throw new RequestException("Provided request is not a valid JSON RPC. The JSON can not be decoded into an array, got: {$decodedType}");
        }

        // Create new requestStore with decoded data.
        $requestStore = new RequestStore($decodedRequest);

        // Create responseStore.
        $responseStore = new ResponseStore();

        // Create JsonRpcController instance.
        $jsonRpcController = JsonRpcController::instance($responseStore);

        // Set requests to run via api.
        $jsonRpcController->setRequestStore($requestStore);

        // Run all requests.
        $jsonRpcController->run();

        // Create new JsonRpcOutput instance.
        $jsonRpcOutput = new JsonRpcOutput($responseStore);

        // Display output to requester.
        $jsonRpcOutput->output(isset($decodedRequest[0]));

        // Expose sessionId by sending x-rpc-header.
        if ($jsonRpcController->getSession() instanceof SessionIdSession) {
            isys_core::send_header(isys_core::HTTP_RPCAuthSession, $jsonRpcController->getSession()
                ->getSessionId());
        }
    }
});

// Run executable in shell.
try {
    $apiEnvironmentShell->run();
} catch (AbstractJsonRpcException $e) {
    jsonRpcErrorOutput($e->getErrorTopic() . $e->getMessage(), $e->getErrorCode());
} catch (\Exception $e) {
    // A shell error has occurred.
    jsonRpcErrorOutput('i-doit System Error: ' . $e->getMessage());
}
