<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Leonard Fischer <lfischer@i-doit.com>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_condition extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods.
     *
     * @var array
     */
    protected $mapping = [
        'isys_obj__id'                   => 'id',
        'isys_obj__title'                => 'title',
        'isys_obj__sysid'                => 'sysid',
        'isys_obj__isys_obj_type__id'    => 'type',
        'isys_obj__created'              => 'created',
        'isys_obj__updated'              => 'updated',
        'isys_obj_type__title'           => ['_L', 'type_title'],
        'isys_obj_type__icon'            => 'type_icon',
        'isys_obj_type_group__id'        => 'type_group',
        'isys_obj_type_group__title'     => ['_L', 'type_group_title'],
        'isys_obj__status'               => 'status',
        'isys_obj__isys_cmdb_status__id' => 'cmdb_status',
        'isys_cmdb_status__title'        => ['_L', 'cmdb_status_title']
    ];

    /**
     * Allowed comparison methods.
     *
     * @var string[]
     */
    private $allowedComparison = ['=', '!=', 'like', 'not like', '>', '>=', '<', '<=', '<>'];

    /**
     * Allowed operators.
     *
     * @var string[]
     */
    private $allowedOperators = ['AND', 'OR'];

    /**
     * Possible options and their parameters.
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Fetches information of the person.
     *
     * @param array $params
     *
     * @return array
     * @throws isys_exception_api
     */
    public function read($params)
    {
        $return = [];

        $conditions = (array)$params['conditions'];

        // Check if the given conditions are well formatted.
        $this->checkConditions($conditions);

        $dao = new isys_cmdb_dao_category_property($this->m_db);

        // @see  API-210  Reset the internal logic to start with a fresh query.
        $dao->reset();

        $queryBuilderConditions = $this->prepareConditionsForQueryBuilding($conditions);
        $sqlAdditions = $dao->create_property_query_condition($queryBuilderConditions);

        // Flatten and filter the resulting array (has a weird nested structure).
        $joins = array_filter((new isys_array($sqlAdditions['joins']))->flatten());

        try {
            // Build our query and retrieve the contents from the DB!
            $sql = 'SELECT obj_main.*, ot.*, cmdb.*
                FROM isys_obj AS obj_main 
                INNER JOIN isys_obj_type AS ot ON isys_obj_type__id = obj_main.isys_obj__isys_obj_type__id
                INNER JOIN isys_cmdb_status AS cmdb ON isys_cmdb_status__id = obj_main.isys_obj__isys_cmdb_status__id
                ' . implode(' ', $joins) . ' 
                WHERE TRUE 
                ' . $sqlAdditions['conditions'] . ';';

            $result = $dao->retrieve($sql);

            while ($row = $result->get_row()) {
                // Don't forget the mapping!
                $return[] = $this->format_by_mapping($this->mapping, $row);
            }
        } catch (Exception $e) {
            throw new isys_exception_api($e->getMessage());
        }

        return $return;
    }

    /**
     * @param array $params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function create($params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function delete($params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function update($params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * Constructor
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;

        parent::__construct();
    }

    /**
     * This method will check the given conditions for compability.
     *
     * @param array $conditions
     *
     * @throws isys_exception_api
     */
    private function checkConditions(array $conditions)
    {
        if (empty($conditions)) {
            throw new isys_exception_api('Please provide at least one condition!');
        }

        foreach ($conditions as $condition) {
            if (!isset($condition['property'])) {
                throw new isys_exception_api('You need to provide a "property" key to every condition.');
            }

            if (!isset($condition['comparison'])) {
                throw new isys_exception_api('You need to provide a "comparison" key to every condition.');
            }

            if (!isset($condition['value'])) {
                throw new isys_exception_api('You need to provide a "value" key to every condition.');
            }

            if (!in_array($condition['comparison'], $this->allowedComparison)) {
                throw new isys_exception_api('You need to provide a valid comparison value. Try one of these: ' . implode(', ', $this->allowedComparison));
            }

            if (isset($condition['operator']) && !in_array($condition['operator'], $this->allowedOperators)) {
                throw new isys_exception_api('You need to provide a valid operator value. Try one of these: ' . implode(', ', $this->allowedOperators));
            }
        }
    }

    /**
     * Prepare the condition-block to work with the given logic.
     *
     * @param array $conditions
     *
     * @return array
     */
    private function prepareConditionsForQueryBuilding(array $conditions)
    {
        $conditionCount = count($conditions);

        foreach ($conditions as $i => &$condition) {
            $condition['category'] = explode('-', $condition['property'])[0];

            if (defined($condition['value'])) {
                $condition['value'] = constant($condition['value']);
            } else {
                $condition['value'] = str_replace('*', '%', str_replace('%', '\%', $condition['value']));
            }

            if (!isset($condition['operator'])) {
                $condition['operator'] = 'AND';
            }

            if (($conditionCount - 1) == $i) {
                unset($condition['operator']);
            }
        }

        return [$conditions];
    }
}
