<?php

/**
 * i-doit
 *
 * API model
 *
 * @package    i-doit
 * @subpackage API
 * @author     Dennis Stücken <dstuecken@synetics.de>
 * @author     Benjamin Heisig <bheisig@synetics.de>
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_cmdb_objects_by_relation extends isys_api_model_cmdb implements isys_api_model_interface
{
    /**
     * Data formatting used in format methods
     *
     * @var array
     */
    protected $m_mapping = [
        'isys_obj__id'                                  => 'id',
        'isys_obj__title'                               => 'title',
        'isys_cmdb_status__id'                          => 'cmdb_status',
        'isys_cmdb_status__title'                       => ['_L', 'cmdb_status_title'],
        'related'                                       => 'related_object',
        'related_title'                                 => 'related_title',
        'related_type'                                  => 'related_type',
        'related_type_title'                            => ['_L', 'related_type_title'],
        'related_cmdb_status'                           => 'related_cmdb_status',
        'related_cmdb_status_title'                     => ['_L', 'related_cmdb_status_title'],
        'isys_catg_relation_list__isys_obj__id__master' => 'master',
        'isys_catg_relation_list__isys_obj__id__slave'  => 'slave',
        'isys_obj_type__id'                             => 'type',
        'isys_obj_type__title'                          => ['_L', 'type_title']
    ];

    /**
     * Possible options and their parameters
     *
     * @var array
     */
    protected $m_options = [
        'read' => []
    ];

    /**
     * Validation
     *
     * @var array
     */
    protected $m_validation = [];

    /**
     * Return objects by relation.
     * [
     *    integer  id             Object identifier
     *    boolean  raw            (optional) Formatting.
     *    mixed    relation_type  (optional) Relation type, may be an integer or string (constant).
     * ]
     *
     * @param array $p_params
     *
     * @return array
     * @throws array
     */
    public function read($p_params)
    {
        // Prepare return array.
        $l_return = [];

        // Prepare filters.
        $l_raw = (bool)$p_params['raw'];
        $l_id = (isset($p_params['id'])) ? $p_params['id'] : null;
        $l_relation_type = (isset($p_params['relation_type'])) ? $p_params['relation_type'] : null;

        // Validate status
        $this->validateStatus($p_params['status']);

        // Check if relation_type is a string constant.
        if (is_string($l_relation_type) && defined($l_relation_type)) {
            $l_relation_type = constant($l_relation_type);
        }

        if ($this->useAuth) {
            isys_auth_cmdb::instance()
                ->check_rights_obj_and_category(isys_auth::VIEW, $l_id, 'C__CATG__RELATION');
        }

        if (!$l_id) {
            throw new isys_exception_api('No object id given. Specify parameter "id" in order to filter objects by relation.');
        }

        // Start looking for objects.
        $l_dao = new isys_cmdb_dao_category_g_relation($this->m_db);
        $l_data = $l_dao->get_related_objects($l_id, $l_relation_type, $p_params['status']);

        // Iterate through result set.
        while ($l_row = $l_data->get_row()) {
            // Fetch more information about the related object:
            $l_related = $this->m_dao->get_object_by_id($l_row['related'])
                ->get_row();

            $l_row['related_title'] = $l_related['isys_obj__title'];
            $l_row['related_type'] = $l_related['isys_obj_type__id'];
            $l_row['related_type_title'] = $l_related['isys_obj_type__title'];
            $l_row['related_cmdb_status_id'] = $l_related['isys_cmdb_status__id'];
            $l_row['related_cmdb_status_title'] = $l_related['isys_cmdb_status__title'];

            $l_key = $l_row['isys_obj__id'];

            if (!$l_raw) {
                $l_row = $this->format_by_mapping($this->m_mapping, $l_row);
            }

            $l_value = [
                'data'     => $l_row,
                'children' => false
            ];

            $l_return[$l_key] = $l_value;
        }

        return $l_return;
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function create($p_params)
    {
        throw new isys_exception_api('Creating is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function delete($p_params)
    {
        throw new isys_exception_api('Deleting is not possible here.');
    }

    /**
     * @param array $p_params Parameters (depends on data method)
     *
     * @throws isys_exception_api
     */
    public function update($p_params)
    {
        throw new isys_exception_api('Updating is not possible here.');
    }

    /**
     * Format by mapping
     *
     * @param array $p_mapping
     * @param array $p_row
     *
     * @return array
     */
    protected function format_by_mapping(array $p_mapping, $p_row)
    {
        // Get mapped result
        $result = parent::format_by_mapping($p_mapping, $p_row);

        // Convert some string values to integers
        $result['id'] = (int)$result['id'];
        $result['related_object'] = (int)$result['related_object'];
        $result['related_type'] = (int)$result['related_type'];
        $result['master'] = (int)$result['master'];
        $result['slave'] = (int)$result['slave'];
        $result['type'] = (int)$result['type'];

        return $result;
    }

    /**
     * isys_api_model_cmdb_objects_by_relation constructor.
     *
     * @param isys_cmdb_dao $p_dao
     */
    public function __construct(isys_cmdb_dao $p_dao)
    {
        $this->m_dao = $p_dao;
        parent::__construct();
    }
}
