<?php

use idoit\Module\Api\Exception\JsonRpc\ParameterException;

/**
 * i-doit APi
 *
 * @package    i-doit
 * @subpackage API
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */
class isys_api_model_idoit_constants implements isys_api_model_interface
{
    private isys_component_template_language_manager $language;
    private isys_cmdb_dao $cmdbDao;

    public function __construct()
    {
        $this->language = isys_application::instance()->container->get('language');
        $this->cmdbDao = isys_application::instance()->container->get('cmdb_dao');
    }

    public function read($params)
    {
        // Mapping of constant groups to method.
        $mapping = [
            'objectTypes'   => 'getObjectTypes',
            'categories'    => 'getCategories',
            'recordStates'  => 'getRecordStates',
            'relationTypes' => 'getRelationTypes',
            'staticObjects' => 'getStaticObjects',
        ];

        // Calculate required constant groups.
        $requiredConstantGroups = $params['constants'] ?: array_keys($mapping);

        if (!is_array($requiredConstantGroups)) {
            throw new ParameterException('Parameter \'constants\' needs to be an array which includes several constant groups.
                                Possible groups are: ' . implode(', ', array_keys($mapping)));
        }

        // Resolved constants store.
        $constants = [];

        // Iterate over constant groups.
        foreach ($requiredConstantGroups as $constantGroup) {
            // Check whether constant group exists and has an resolver method.
            if (!method_exists($this, $mapping[$constantGroup])) {
                continue;
            }

            // Resolve constant groups constants.
            $constants[$constantGroup] = call_user_func_array([$this, $mapping[$constantGroup]], []);
        }

        return $constants;
    }

    private function getObjectTypes(): array
    {
        $constants = [];
        $result = $this->cmdbDao->get_object_types();

        while ($row = $result->get_row()) {
            $constants[$row['isys_obj_type__const']] = $this->language->get($row['isys_obj_type__title']);
        }

        return $constants;
    }

    private function getCategories(): array
    {
        $constants = [];

        foreach (['g', 's', 'g_custom'] as $categoryType) {
            // Set category type specific resource
            $result = $this->cmdbDao->get_isysgui('isysgui_cat' . $categoryType);

            while ($row = $result->get_row()) {
                $constants[$categoryType][$row['isysgui_cat' . $categoryType . '__const']] = $this->language->get($row['isysgui_cat' . $categoryType . '__title']);
            }
        }

        return $constants;
    }

    private function getStaticObjects(): array
    {
        $constants = [];
        $result = $this->cmdbDao->retrieve('SELECT * FROM isys_obj WHERE isys_obj__const <> \'\';');

        while ($row = $result->get_row()) {
            $constants[$row['isys_obj__const']] = $this->language->get($row['isys_obj__title']);
        }

        return $constants;
    }

    private function getRelationTypes(): array
    {
        $constants = [];
        $result = $this->cmdbDao->retrieve('SELECT * FROM isys_relation_type WHERE TRUE;');

        while ($row = $result->get_row()) {
            $constants[$row['isys_relation_type__const']] = $this->language->get($row['isys_relation_type__title']);
        }

        return $constants;
    }

    private function getRecordStates(): array
    {
        return [
            'C__RECORD_STATUS__NORMAL'   => $this->language->get('LC__CMDB__RECORD_STATUS__NORMAL'),
            'C__RECORD_STATUS__ARCHIVED' => $this->language->get('LC__CMDB__RECORD_STATUS__ARCHIVED'),
            'C__RECORD_STATUS__DELETED'  => $this->language->get('LC__CMDB__RECORD_STATUS__DELETED'),
            'C__RECORD_STATUS__PURGE'    => $this->language->get('LC__CMDB__RECORD_STATUS__PURGE'),
        ];
    }
}
