<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api;

use idoit\Console\IdoitConsoleApplication;
use Symfony\Component\Console\Input\ArrayInput;
use Symfony\Component\Console\Output\BufferedOutput;

/**
 * Class ApiCommandRunner
 *
 * @package idoit\Module\Api
 */
class ApiCommandRunner
{
    /**
     * @var string
     */
    protected $commandName;

    /**
     * @var int
     */
    protected $exitCode;

    /** @var BufferedOutput */
    protected BufferedOutput $output;

    /**
     * @var string
     */
    protected $idoitRootDirectoryPath;

    /**
     * Options
     *
     * @var array
     */
    protected $options = [];

    /**
     * Arguments
     *
     * @var array
     */
    protected $arguments = [];

    /**
     * Add an new option
     *
     * @param string $optionName
     * @param string $optionValue
     */
    public function addOption($optionName, $optionValue)
    {
        $this->options[(strlen($optionName) == 1 ? '-' : '--') . $optionName] = $optionValue;
    }

    /**
     * Add an new argument
     *
     * @param string $argumentName
     * @param string $argumentValue
     */
    public function addArgument($argumentName, $argumentValue)
    {
        $this->arguments[$argumentName] = $argumentValue;
    }

    /**
     * @return int
     */
    public function getExitCode()
    {
        return $this->exitCode;
    }

    /**
     * @return array
     */
    private function getArguments()
    {
        return $this->arguments;
    }

    /**
     * @return array
     */
    private function getOptions()
    {
        return $this->options;
    }

    /**
     * @return string
     */
    private function getCommandName()
    {
        return $this->commandName;
    }

    /**
     * @param string $commandName
     *
     * @return ApiCommandRunner
     */
    private function setCommandName($commandName)
    {
        $this->commandName = $commandName;

        return $this;
    }

    /**
     * @param int $exitCode
     *
     * @return ApiCommandRunner
     */
    private function setExitCode($exitCode)
    {
        $this->exitCode = $exitCode;

        return $this;
    }

    /**
     * @return array
     */
    public function getCommandOutput(): array
    {
        return explode(PHP_EOL, $this->output->fetch());
    }


    /**
     * Check whether execution was successful
     *
     * @return bool
     */
    public function success()
    {
        return $this->getExitCode() === 0;
    }

    /**
     * Build command input
     *
     * @param $commandName
     * @param $options
     * @param $arguments
     *
     * @return array
     */
    private function buildCommandInput($commandName, $options, $arguments)
    {
        return [
                'command' => $commandName,
            ] + $options + $arguments;
    }

    /**
     * Run command
     *
     * @return int
     * @throws \Exception
     */
    public function run()
    {
        // Check whether idoit root directory path exists or not
        if (!file_exists($this->getIdoitRootDirectoryPath())) {
            throw new \Exception('Provided i-doit root directory path does not exist.');
        }

        // Get working directory
        $currentWorkingDirectory = getcwd();

        // Change working folder to i-doit root directory
        chdir($this->getIdoitRootDirectoryPath());

        // Create and configure idoit console application
        $application = new IdoitConsoleApplication();
        $application->setAutoExit(false);

        // Create input for command
        $input = new ArrayInput($this->buildCommandInput($this->getCommandName(), $this->getOptions(), $this->getArguments()));

        // Run command
        $exitCode = $application->run($input, $this->output);

        // Set command exit code
        $this->setExitCode($exitCode);

        // Change working directory back
        if ($currentWorkingDirectory !== false) {
            chdir($currentWorkingDirectory);
        }

        return $exitCode;
    }

    /**
     * Get idoit root directory path
     *
     * @return string
     */
    private function getIdoitRootDirectoryPath()
    {
        return $this->idoitRootDirectoryPath;
    }

    /**
     * Set idoit root directory path
     *
     * @param string $idoitRootDirectoryPath
     *
     * @return ApiCommandRunner
     */
    private function setIdoitRootDirectoryPath($idoitRootDirectoryPath)
    {
        $this->idoitRootDirectoryPath = $idoitRootDirectoryPath;

        return $this;
    }

    /**
     * ApiCommandRunner constructor.
     *
     * @param string $idoitRootDirectoryPath
     * @param string $commandName
     */
    public function __construct($idoitRootDirectoryPath, $commandName)
    {
        $this->setIdoitRootDirectoryPath($idoitRootDirectoryPath)
            ->setCommandName($commandName);

        $this->output = new BufferedOutput();
    }
}
