<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Dialog;

use idoit\Component\Property\Configuration\PropertyDependency;
use idoit\Component\Property\Property;

/**
 * Class DialogPlusAdmin
 *
 * @package idoit\Module\Api
 */
class DialogPlusAdmin extends DialogAdmin
{
    /**
     * Find or create id for provided value
     *
     * @param string $value
     *
     * @return false|int|null|string
     * @throws \isys_exception_dao
     * @throws \Exception
     */
    public function findOrCreate($value)
    {
        if ($this->valueExists($value)) {
            return (int)$this->getIdByValue($value);
        }

        if (empty($value)) {
            return null;
        }

        if ($this->getMode() === self::MODE_ARRAY && $this->isCustomDialog()) {
            return $this->createCustomDialog($value);
        }

        if ($this->getMode() === self::MODE_TABLE) {
            $propertyDefinition = $this->getPropertyDefinition();

            // Property require some other data - Skip further processing
            if (!empty($propertyDefinition[C__PROPERTY__FORMAT__REQUIRES])) {
                return null;
            }

            // @see  API-223  Property is dependent to other data - Skip further processing.
            if (isset($propertyDefinition[C__PROPERTY__DEPENDENCY])) {
                // @see  API-283  Check if we are handling a 'PropertyFactory' definition.
                if ($propertyDefinition[C__PROPERTY__DEPENDENCY] instanceof PropertyDependency) {
                    // Only jump out, if the array does not eval to 'empty' (= false).
                    if ($propertyDefinition[C__PROPERTY__DEPENDENCY]->toArray()) {
                        return null;
                    }
                } elseif ($propertyDefinition[C__PROPERTY__DEPENDENCY] &&
                    $propertyDefinition[C__PROPERTY__DEPENDENCY]['type'] !== 'primary'
                ) {
                    return null;
                }
            }

            $daoInstance = \isys_application::instance()->container->get('cmdb_dao');

            // Get table name
            $tableName = $propertyDefinition[C__PROPERTY__UI]['params']['p_strTable'] ?: $propertyDefinition[C__PROPERTY__DATA][C__PROPERTY__DATA__REFERENCES][0];

            // Create insert statement
            if ($daoInstance->fieldsExistsInTable($tableName, [$tableName . '__identifier'])) {
                $sql = 'INSERT INTO ' . $tableName . '
                    SET ' . $tableName . '__title = ' . $daoInstance->convert_sql_text($value) . ',
                    ' . $tableName . '__status = ' . $daoInstance->convert_sql_int(C__RECORD_STATUS__NORMAL) . ',
                    ' . $tableName . '__identifier = ' . $daoInstance->convert_sql_text($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['identifier']) . ';';
            } else {
                $sql = 'INSERT INTO ' . $tableName . '
                    SET ' . $tableName . '__title = ' . $daoInstance->convert_sql_text($value) . ',
                    ' . $tableName . '__status = ' . $daoInstance->convert_sql_int(C__RECORD_STATUS__NORMAL) . ';';
            }

            // Create dialog entry and return its id
            if ($daoInstance->update($sql) && $daoInstance->apply_update()) {
                return (int)$daoInstance->get_last_insert_id();
            }
        }

        return null;
    }

    /**
     * @return bool
     */
    public function isCustomDialog()
    {
        return isset($this->getPropertyDefinition()[Property::C__PROPERTY__DATA][Property::C__PROPERTY__DATA__REFERENCES])
            && $this->getPropertyDefinition()[Property::C__PROPERTY__DATA][Property::C__PROPERTY__DATA__REFERENCES][0] === 'isys_dialog_plus_custom';
    }

    /**
     * @param string $value
     *
     * @return int|null
     * @throws \Exception
     */
    public function createCustomDialog(string $value): ?int
    {
        $daoInstance = \isys_application::instance()->container->get('cmdb_dao');
        $identifier = $this->getPropertyDefinition()[Property::C__PROPERTY__UI][Property::C__PROPERTY__UI__PARAMS]['identifier'];

        $sql = "INSERT INTO isys_dialog_plus_custom SET
            isys_dialog_plus_custom__title = {$daoInstance->convert_sql_text($value)},
            isys_dialog_plus_custom__status = {$daoInstance->convert_sql_int(C__RECORD_STATUS__NORMAL)},
            isys_dialog_plus_custom__identifier = {$daoInstance->convert_sql_text($identifier)}";

        if ($daoInstance->update($sql) && $daoInstance->apply_update()) {
            return (int)$daoInstance->get_last_insert_id();
        }
        return null;
    }
}
