<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\Category;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Module\Api\Category\Entry;
use idoit\Module\Api\Endpoint\v2\Cmdb\AbstractCmdbEndpoint;
use idoit\Module\Api\Endpoint\Validate;
use idoit\Module\Api\Exception\JsonRpc\AuthenticationException;
use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use isys_auth;
use isys_module_cmdb;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB category delete endpoint (v2).
 *
 * @see       API-484
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Delete extends AbstractCmdbEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition('cmdb.category.delete.v2', 'LC__ADDON__API__ENDPOINT__CMDB_CATEGORY_DELETE', [
            new RequiredParameter('object', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__OBJECT_ID', [Validate::class, 'isId']),
            new RequiredParameter('category', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CATEGORY_CONSTANT', fn ($category) => Validate::isConstant($category, 'C__CAT') && defined($category)),
            new RequiredParameter('entry', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__ID', [Validate::class, 'isId']),
        ]);
    }

    public function request(Request $request): JsonRpcResponse
    {
        $objectId = $request->request->get('object');
        $categoryConstant = $request->request->get('category');
        $entryId = $request->request->get('entry');

        // Create category entry instance
        $categoryEntry = new Entry($objectId, $categoryConstant, $entryId);

        // Check rights
        $isAllowedToDelete = isys_module_cmdb::getAuth()->has_rights_in_obj_and_category(isys_auth::DELETE, $objectId, $categoryConstant);

        if (!$isAllowedToDelete) {
            throw new AuthenticationException('You are not allowed to delete this category entry.');
        }

        // @see API-191 Check and react to the object lock.
        $this->lockObject($objectId);

        if (!$categoryEntry->delete()) {
            throw new InternalErrorException('Unable to delete category entry.');
        }

        // @see API-191 Unlock the object once the request has finished.
        $this->unlockObject($objectId);

        return new JsonRpcResponse([
            'success' => true,
            'message' => "Entry {$entryId} has been successfully deleted."
        ]);
    }
}
