<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectType;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Component\Processor\Dto\ObjectType\Dto;
use idoit\Component\Processor\Dto\ObjectType\ReadRequest;
use idoit\Component\Processor\Exception\AuthorizationException;
use idoit\Component\Processor\Exception\InternalSystemException;
use idoit\Component\Processor\Exception\ValidationException;
use Idoit\Dto\Serialization\Serializer;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use isys_exception_database;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type read endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractObjectTypeEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type.read.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_READ',
            [
                new OptionalParameter('id', [Parameter::TYPE_INTEGER, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__ID_OR_ARRAY', [Validate::class, 'isIdOrArrayOfIds']),
                new OptionalParameter('constant', [Parameter::TYPE_STRING, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__CONSTANT_OR_ARRAY', fn (string|array $constants) => Validate::isConstantOrArrayOfConstants($constants, 'C__OBJTYPE__', true)),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__IS_VISIBLE'),
                new OptionalParameter('with-categories', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__WITH_CATEGORIES', null, false),
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type.read.v2'),
                'With ids' => Helper::exampleRequest('cmdb.object-type.read.v2', ['id' => 123]),
                'With constants' => Helper::exampleRequest('cmdb.object-type.read.v2', ['constant' => ['C__OBJTYPE__SERVER', 'C__OBJTYPE__CLIENT']]),
            ],
        );
    }

    /**
     * @param Request $request
     * @return JsonRpcResponse
     * @throws AuthorizationException
     * @throws InternalSystemException
     * @throws ValidationException
     * @throws isys_exception_database
     */
    public function request(Request $request): JsonRpcResponse
    {
        $objectTypes = [];

        // @see API-598 Refactor usages of 'get' if the values can contain arrays.
        $parameters = $request->request->all();
        $status = $request->request->get('visible') !== null
            ? [($request->request->get('visible') ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH)]
            : [];

        $objectTypeIds = array_unique(array_merge(
            (array)$parameters['id'] ?? [],
            array_map('constant', (array)$parameters['constant'] ?? [])
        ));

        $response = $this->objectTypeProcessor->read(new ReadRequest($objectTypeIds, $status, $parameters['with-categories']));

        foreach ($response->entries as $dto) {
            /** @var Dto $dto */
            $objectType = Serializer::toJson($dto);

            $objectTypes[] = $objectType;
        }

        return new JsonRpcResponse($objectTypes);
    }
}
