<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectType;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Component\Processor\Dto\ObjectType\UpdateRequest;
use idoit\Component\Processor\Exception\AuthorizationException;
use idoit\Component\Processor\Exception\InternalSystemException;
use idoit\Component\Processor\Exception\ValidationException;
use Idoit\Dto\Serialization\Serializer;
use idoit\Module\Api\Endpoint\Validate;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type update (by constant) endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class UpdateByConstant extends AbstractObjectTypeEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type.update-by-constant.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_UPDATE',
            [
                new RequiredParameter('constant', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CONSTANT', fn (string $constant) => Validate::isConstant($constant, 'C__OBJTYPE__', true)),
                new OptionalParameter('title', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__TITLE', fn ($title) => trim($title) !== ''),
                new OptionalParameter('objectTypeGroup', [Parameter::TYPE_INTEGER, Parameter::TYPE_STRING], 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__OBJECT_TYPE_GROUP', fn (int|string $identifier) => is_int($identifier) ? Validate::isId($identifier) : Validate::isConstant($identifier, 'C__OBJTYPE_GROUP__', true)),
                new OptionalParameter('description', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__OBJECT_TYPE_GROUP'),
                new OptionalParameter('specificCategory', [Parameter::TYPE_INTEGER, Parameter::TYPE_STRING], 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__SPECIFIC_CATEGORY', fn (int|string $identifier) => is_int($identifier) || Validate::isConstant($identifier, 'C__CATS__', true)),
                new OptionalParameter('isContainer', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__IS_CONTAINER', null, false),
                new OptionalParameter('isPositionableInRack', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__IS_POSITIONABLE_IN_RACK', null, false),
                new OptionalParameter('isVisible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__IS_VISIBLE', null, true),
                new OptionalParameter('isRelationMaster', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__IS_RELATION_MASTER', null, false),
                new OptionalParameter('hasOverviewPage', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__HAS_OVERVIEW_PAGE', null, false),
                new OptionalParameter('sort', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__SORT', null, defined_or_default('C__PROPERTY__DEFAULT_SORT', 65535)),
                new OptionalParameter('color', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__COLOR', null, '#ffffff'),
                new OptionalParameter('defaultTemplate', [Parameter::TYPE_INTEGER, Parameter::TYPE_STRING], 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__DEFAULT_TEMPLATE'),
                new OptionalParameter('sysidPrefix', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE__SYSID_PREFIX'),
            ],
        );
    }

    /**
     * @param Request $request
     * @return JsonRpcResponse
     * @throws AuthorizationException
     * @throws InternalSystemException
     * @throws ValidationException
     */
    public function request(Request $request): JsonRpcResponse
    {
        $parameters = array_filter($request->request->all(), fn ($val) => $val !== null);
        $constant = $parameters['constant'];
        unset($parameters['constant']);

        $parameters['id'] = (int)constant($constant);

        $updateResponse = $this->objectTypeProcessor->update(Serializer::fromJson(UpdateRequest::class, $parameters));

        return new JsonRpcResponse([
            'success'  => true,
            'id'       => $updateResponse->id,
            'constant' => $constant,
            'message'  => "Object type #{$updateResponse->id} successfully updated."
        ]);
    }
}
