<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Component\Processor\Dto\ObjectTypeGroup\CreateRequest;
use idoit\Component\Processor\Exception\AuthorizationException;
use idoit\Component\Processor\Exception\InternalSystemException;
use idoit\Component\Processor\Exception\ValidationException;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use isys_auth;
use isys_exception_auth;
use isys_module_system;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group create endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Create extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type-group.create.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_CREATE',
            [
                new RequiredParameter('title', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_TITLE', fn ($title) => trim($title) !== ''),
                new OptionalParameter('constant', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_CONSTANT', fn ($constant) => Validate::isConstant($constant, 'C__OBJTYPE_GROUP__SD_')),
                new OptionalParameter('sort', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_SORT', null, defined_or_default('C__PROPERTY__DEFAULT_SORT', 65535)),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_VISIBLE', null, true),
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type-group.create.v2', [
                    'title' => 'Example group',
                ]),
                'Full request' => Helper::exampleRequest('cmdb.object-type-group.create.v2', [
                    'title' => 'Example group',
                    'constant' => 'C__OBJTYPE_GROUP__SD_EXAMPLE_GROUP',
                    'visible' => true,
                    'sort' => 10,
                ]),
            ],
            [
                'Basic response' => Helper::exampleResponse([
                    'success' => true,
                    'id' => 123,
                    'message' => "Object type group 'Example group' (#123) successfully created.",
                ]),
            ],
        );
    }

    /**
     * @param Request $request
     * @return JsonRpcResponse
     * @throws AuthorizationException
     * @throws InternalSystemException
     * @throws ValidationException
     * @throws isys_exception_auth
     */
    public function request(Request $request): JsonRpcResponse
    {
        // @todo Get rid of specific AUTH check, after it was implemented in processor.
        isys_module_system::getAuth()->check(isys_auth::CREATE, 'globalsettings/qcw');

        $title = $request->request->get('title');

        $response = $this->objectTypeGroupProcessor->create(new CreateRequest(
            $title,
            $request->request->get('constant'),
            $request->request->get('sort'),
            $request->request->get('visible') ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH
        ));

        return new JsonRpcResponse([
            'success' => true,
            'id' => $response->id,
            'message' => "Object type group '{$title}' (#{$response->id}) successfully created."
        ]);
    }
}
