<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Component\Processor\Dto\ObjectTypeGroup\Dto;
use idoit\Component\Processor\Dto\ObjectTypeGroup\ReadRequest;
use idoit\Component\Processor\Exception\AuthorizationException;
use idoit\Component\Processor\Exception\InternalSystemException;
use idoit\Component\Processor\Exception\ValidationException;
use Idoit\Dto\Serialization\Serializer;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use isys_exception_database;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group read endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Read extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type-group.read.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_READ',
            [
                new OptionalParameter('id', [Parameter::TYPE_INTEGER, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__ID_OR_ARRAY', [Validate::class, 'isIdOrArrayOfIds']),
                new OptionalParameter('constant', [Parameter::TYPE_STRING, Parameter::TYPE_ARRAY], 'LC__ADDON__API__PARAMETER__CONSTANT_OR_ARRAY', fn (string|array $constants) => Validate::isConstantOrArrayOfConstants($constants, 'C__OBJTYPE_GROUP__', true)),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_VISIBLE'),
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type-group.read.v2'),
                'With ids' => Helper::exampleRequest('cmdb.object-type-group.read.v2', ['id' => 1]),
                'With constants' => Helper::exampleRequest('cmdb.object-type-group.read.v2', ['constant' => 'C__OBJTYPE_GROUP__SOFTWARE']),
            ],
            ['Basic response' => Helper::exampleResponse([
                [
                    'id'        => 1,
                    'constant'  => 'C__OBJTYPE_GROUP__SOFTWARE',
                    'title'     => 'Software',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__SOFTWARE',
                    'sort'      => 1,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                [
                    'id'        => 2,
                    'constant'  => 'C__OBJTYPE_GROUP__INFRASTRUCTURE',
                    'title'     => 'Infrastructure',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__INFRASTRUCTURE',
                    'sort'      => 2,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                [
                    'id'        => 3,
                    'constant'  => 'C__OBJTYPE_GROUP__OTHER',
                    'title'     => 'Other',
                    'titleRaw'  => 'LC__CMDB__OBJTYPE_GROUP__OTHER',
                    'sort'      => 3,
                    'status'    => 2,
                    'visible'   => true,
                    'deletable' => false
                ],
                ['...']
            ])
            ]
        );
    }

    /**
     * @param Request $request
     * @return JsonRpcResponse
     * @throws AuthorizationException
     * @throws InternalSystemException
     * @throws ValidationException
     * @throws isys_exception_database
     */
    public function request(Request $request): JsonRpcResponse
    {
        $objectTypeGroups = [];

        // @see API-598 Refactor usages of 'get' if the values can contain arrays.
        $parameters = $request->request->all();
        $id = array_unique((array)($parameters['id'] ?? []));
        $constant = array_unique((array)($parameters['constant'] ?? []));
        $status = $request->request->get('visible') !== null
            ? ($request->request->get('visible') ? C__RECORD_STATUS__NORMAL : C__RECORD_STATUS__BIRTH)
            : null;

        // @todo Use 'ReadRequest' as soon as constants can be used.
        $ids = [];
        $result = $this->read($id, $constant, $status);

        while ($row = $result->get_row()) {
            $ids[] = (int)$row['isys_obj_type_group__id'];
        }

        $response = $this->objectTypeGroupProcessor->read(new ReadRequest($ids));

        foreach ($response->entries as $dto) {
            /** @var Dto $dto */
            $objectTypeGroup = Serializer::toJson($dto);

            // Populate some convenience props.
            $objectTypeGroup['visible'] = $dto->status->id == C__RECORD_STATUS__NORMAL;
            $objectTypeGroup['deletable'] = str_starts_with($dto->constant, 'C__OBJTYPE_GROUP__SD_');

            $objectTypeGroups[] = $objectTypeGroup;
        }

        return new JsonRpcResponse($objectTypeGroups);
    }
}
