<?php

namespace idoit\Module\Api\Endpoint\v2\Cmdb\ObjectTypeGroup;

use idoit\Api\EndpointDefinition;
use idoit\Api\JsonRpcResponse;
use idoit\Api\Parameter\OptionalParameter;
use idoit\Api\Parameter\Parameter;
use idoit\Api\Parameter\RequiredParameter;
use idoit\Component\Processor\Dto\ObjectTypeGroup\UpdateRequest;
use idoit\Component\Processor\Exception\AuthorizationException;
use idoit\Component\Processor\Exception\InternalSystemException;
use idoit\Component\Processor\Exception\ValidationException;
use idoit\Module\Api\Endpoint\Helper;
use idoit\Module\Api\Endpoint\Validate;
use isys_auth;
use isys_exception_auth;
use isys_module_system;
use Symfony\Component\HttpFoundation\Request;

/**
 * CMDB object type group update endpoint (v2).
 *
 * @package   idoit\Api
 * @copyright synetics GmbH
 * @license   http://www.i-doit.com/license
 */
class Update extends AbstractObjectTypeGroupEndpoint
{
    public function getDefinition(): EndpointDefinition
    {
        return new EndpointDefinition(
            'cmdb.object-type-group.update.v2',
            'LC__ADDON__API__ENDPOINT__CMDB_OBJECT_TYPE_GROUP_UPDATE',
            [
                new RequiredParameter('id', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__ID', [Validate::class, 'isId']),
                new OptionalParameter('title', Parameter::TYPE_STRING, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_TITLE', fn ($title) => trim($title) !== ''),
                new OptionalParameter('sort', Parameter::TYPE_INTEGER, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_SORT'),
                new OptionalParameter('visible', Parameter::TYPE_BOOLEAN, 'LC__ADDON__API__PARAMETER__CMDB_OBJECT_TYPE_GROUP_VISIBLE')
            ],
            [
                'Basic request' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'title' => 'New object type group name']),
                'Change visibility' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'visible' => false]),
                'Change order' => Helper::exampleRequest('cmdb.object-type-group.update.v2', ['id' => 1234, 'sort' => 99]),
            ],
            ['Basic response' => Helper::exampleResponse(['success' => true, 'id' => 1234, 'message' => 'Object type group #1234 successfully updated.'])]
        );
    }

    /**
     * @param Request $request
     * @return JsonRpcResponse
     * @throws AuthorizationException
     * @throws InternalSystemException
     * @throws ValidationException
     * @throws isys_exception_auth
     */
    public function request(Request $request): JsonRpcResponse
    {
        // @todo Get rid of specific AUTH check, after it was implemented in processor.
        isys_module_system::getAuth()->check(isys_auth::EDIT, 'globalsettings/qcw');

        $id = $request->request->get('id');

        $visible = $request->request->get('visible') !== null
            ? ($request->request->get('visible') ? C__RECORD_STATUS__BIRTH : C__RECORD_STATUS__NORMAL)
            : null;

        $this->objectTypeGroupProcessor->update(new UpdateRequest(
            $id,
            $request->request->get('title'),
            $request->request->get('sort'),
            $visible
        ));

        return new JsonRpcResponse([
            'success' => true,
            'id'      => $id,
            'message' => "Object type group #{$id} successfully updated."
        ]);
    }
}
