<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Oscar Pohl <opohl@i-doit.com>
 * @version    1.15
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use Exception;
use idoit\Exception\JsonException;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_application;
use isys_cmdb_dao_category_g_version;
use isys_exception_cmdb;
use isys_exception_dao;

/**
 * Class ModelProcessor
 *
 * @package idoit\Module\Api\Model\Cmdb\Category\Processor
 */
class ApplicationAssignedObjProcessor extends AbstractCategoryProcessor implements SyncModifier, RequestModifier
{
    /**
     * @var array
     */
    protected array $rawRequest = [];

    /**
     * @var isys_cmdb_dao_category_g_version
     */
    private isys_cmdb_dao_category_g_version $versionDao;

    /**
     * @return array
     */
    public function getAffectedPropertiesByRequest()
    {
        return [
            'assigned_version',
        ];
    }

    public function getAffectedPropertiesBySync()
    {
        return [
            'assigned_database_schema',
            'assigned_it_service',
        ];
    }

    /**
     * Modify api request
     *
     * @param array $request
     *
     * @return array
     * @throws JsonException
     * @throws isys_exception_cmdb
     * @throws isys_exception_dao
     */
    public function modifyRequest(array $request)
    {
        $this->versionDao = isys_cmdb_dao_category_g_version::instance(isys_application::instance()->container->get('database'));
        $isCreationRequest = in_array($request['option'], ['save', 'update', 'create'], true);

        // Check whether request wants to create some data and application is set
        if ($request['option'] === 'create' && isset($request['data']['assigned_database_schema'])) {
            // Create application  dao
            $dao = new \isys_cmdb_dao_category_s_application_assigned_obj(\isys_application::instance()->container->get('database'));

            // Create application  assignment
            $entryId = $dao->create($request['objID'], C__RECORD_STATUS__NORMAL, $request['data']['object'], '');

            // Transform request into an update
            $request['option'] = 'update';
            $request['id'] = $entryId;
            $request['data']['id'] = $entryId;
        }

        $this->rawRequest = $request;

        // @see ID-9329 Check if 'assigned_version' was passed and if it makes sense.
        if ($isCreationRequest && isset($request['objID']) && $request['objID'] > 0 && isset($request['data']['assigned_version'])) {
            $versionId = $this->versionExists((int)$request['objID'], $request['data']['assigned_version']);

            if ($versionId === null) {
                if (is_string($request['data']['assigned_version'])) {
                    $request['data']['assigned_version'] = $this->createVersion((int)$request['objID'], $request['data']['assigned_version']);
                }
            } else {
                $request['data']['assigned_version'] = $versionId;
            }
        }

        return $request;
    }

    /**
     * @param int    $applicationObjectId
     * @param string $version
     *
     * @return int|null
     */
    private function createVersion(int $applicationObjectId, string $version): ?int
    {
        $id = $this->versionDao->create_data(['isys_obj__id' => $applicationObjectId, 'title' => $version]);

        if (is_int($id)) {
            return $id;
        }

        return null;
    }

    /**
     * @param int        $applicationObjectId
     * @param int|string $version
     *
     * @return int|null
     * @throws Exception
     */
    private function versionExists(int $applicationObjectId, $version): ?int
    {
        $result = $this->versionDao->get_data(null, $applicationObjectId);

        while ($row = $result->get_row()) {
            if (is_int($version) && $version == $row['isys_catg_version_list__id']) {
                return (int)$row['isys_catg_version_list__id'];
            }

            if (is_string($version) && $version == $row['isys_catg_version_list__title']) {
                return (int)$row['isys_catg_version_list__id'];
            }
        }

        return null;
    }

    /**
     * Modify sync data
     *
     * @param array $syncData
     *
     * @return array
     */
    public function modifySyncData(array $syncData)
    {
        /**
         * Check whether assigned database schema was setted explicitly by request
         * or was added by isys_api_model_cmdb_category::__to_sync_structure().
         *
         * In last cas we will remove it because the key will simply hold the
         * isys_catg_relation__id but sync will interpret it as object id.
         */
        if (is_array($this->rawRequest['data']) && !array_key_exists('assigned_database_schema', $this->rawRequest['data'])) {
            $syncData['properties']['assigned_database_schema'] = null;
        }
        if (is_array($this->rawRequest['data']) && !array_key_exists('assigned_it_service', $this->rawRequest['data'])) {
            $syncData['properties']['assigned_it_service'] = null;
        }

        return $syncData;
    }
}
