<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Model\Cmdb\Category\Processor;

use idoit\Module\Api\Dialog\DialogPlusAdmin;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\RequestModifier;
use idoit\Module\Api\Model\Cmdb\Category\Processor\Provider\SyncModifier;
use isys_exception_api_validation;
use isys_export_data;
use isys_export_helper;

/**
 * IpAddressesProcessor
 *
 * @see        API-263  In order to provide the correct data structure, we implemented this processor.
 * @package    idoit\Module\Api\Model\Category
 */
class Layer2NetProcessor extends AbstractCategoryProcessor implements RequestModifier, SyncModifier
{
    /**
     * @var array|null
     */
    private $ipHelper;

    /**
     * @return array
     */
    public function getAffectedPropertiesByRequest()
    {
        return [
            'ip_helper_addresses',

        ];
    }

    /**
     * @return array
     */
    public function getAffectedPropertiesBySync()
    {
        return [
            'ip_helper_addresses',
        ];
    }

    /**
     * @param array $request
     *
     * @return array
     * @throws isys_exception_api_validation
     */
    public function modifyRequest(array $request)
    {
        $this->ipHelper = null;

        // Only process writing actions.
        if (!in_array($request['option'], ['save', 'update', 'create'], true)) {
            return $request;
        }

        // If the 'ip_helper_addresses' property is not set, return the data unchanged.
        if (!isset($request['data']['ip_helper_addresses']) || empty($request['data']['ip_helper_addresses'])) {
            return $request;
        }

        // Otherwise force it to be an array.
        $request['data']['ip_helper_addresses'] = (array)$request['data']['ip_helper_addresses'];

        // We'll validate the provided data.
        foreach ($request['data']['ip_helper_addresses'] as $address) {
            if (!is_array($address)) {
                throw new isys_exception_api_validation('The provided "ip_helper_addresses" data needs to be an array of arrays containing the keys "ip" and "type_title".', ['ip_helper_addresses' => '[]']);
            }

            // Check if the provided addresses contain the correct keys.
            if (!isset($address['ip'])) {
                throw new isys_exception_api_validation('The "ip_helper_addresses" array entry needs to contain the "ip" key and value.', ['ip_helper_addresses' => '[ip: "...", type_title: ""]']);
            }

            if (!isset($address['type_title'])) {
                throw new isys_exception_api_validation('The "ip_helper_addresses" array entry needs to contain the "type_title" key and value.', ['ip_helper_addresses' => '[ip: "...", type_title: ""]']);
            }
        }

        // Write the ip-helper to a local variable to retrieve it later (and bypass validation).
        $this->ipHelper = $request['data']['ip_helper_addresses'];
        unset($request['data']['ip_helper_addresses']);

        return $request;
    }

    /**
     * @param array $syncData
     *
     * @return array
     * @throws \isys_exception_database
     */
    public function modifySyncData(array $syncData)
    {
        // @see API-632 Only apply the logic from API-604 if no assignments where passed.
        if (isset($syncData['data_id']) && !isset($syncData['properties']['layer3_assignments'][C__DATA__VALUE])) {
            $layer3Assignments = (new isys_export_helper())->layer_3_assignment($syncData['data_id']);

            if ($layer3Assignments instanceof isys_export_data) {
                $data = $layer3Assignments->get_data();
                if (!empty($data)) {
                    $newData = [];

                    foreach ($data as $item) {
                        $item['type'] = $this->getDao()->get_objTypeID($item['id']);
                        $newData[] = $item;
                    }
                    $syncData['properties']['layer3_assignments'][C__DATA__VALUE] = $newData;
                }
            }
        }

        // If the 'IP helper' property is not set, return the data unchanged.
        if ($this->ipHelper === null) {
            return $syncData;
        }

        $properties = $this->getDao()->get_properties();

        $dialogAdmin = DialogPlusAdmin::byTableName('isys_layer2_iphelper_type', $properties['ip_helper_addresses']);

        // We'll validate the provided data.
        foreach ($this->ipHelper as &$address) {
            if (is_int($address['type_title'])) {
                // In the case of an passed integer, we'll get the corresponding title (int = entry ID).
                $address['type_title'] = $dialogAdmin->getValueById($address['type_title']);
            }
        }

        $syncData['properties']['ip_helper_addresses'][C__DATA__VALUE] = $this->ipHelper;

        return $syncData;
    }
}
