<?php

namespace idoit\Module\Api\Model\External\CategoryStrategy;

use idoit\Module\Api\Model\External\Logger;
use idoit\Module\Api\Model\External\Push\Entry;
use idoit\Module\Api\Model\External\Handler\Category;
use idoit\Module\Api\Model\External\Handler\CiObject;
use idoit\Module\Cmdb\Component\SyncMerger\Config;
use idoit\Module\Cmdb\Component\SyncMerger\Merger;
use isys_cmdb_dao_category;
use isys_import_handler_cmdb;

class Update extends AbstractCategoryStrategy implements CategoryStrategyInterface
{
    protected const STRATEGY = 'update';

    /**
     * @param isys_cmdb_dao_category $dao
     * @param array                  $entries
     *
     * @return array
     */
    public function prepare(isys_cmdb_dao_category $dao, CiObject $ciObject, array $entries): array
    {
        $logger = Logger::instance();
        if (!count($entries)) {
            $logger->info('Nothing to proceed');
            return [];
        }

        if ($dao->is_multivalued()) {
            $logger->debug("Handling multi value category.");
            return $this->prepareMultivalue($dao, $ciObject, $entries);
        }
        $logger->debug("Handling single value category.");

        $entry = current($entries);
        $externalType = "{$ciObject->getExtType()}/{$ciObject->getExtId()}/{$dao->get_category_const()}";
        $externalId = key($entries);

        $categoryEntry = new Category($ciObject, $dao, $externalType, $dao->get_category_const());
        $refId = $categoryEntry->findRefIdByExternalId($externalId);
        $categoryEntryId = $categoryEntry->findSingleValueEntryId();
        $created = false;

        if ($refId && $categoryEntryId && $refId === $categoryEntryId) {
            $logger->info("Category entry for external id {$externalType} exists. Using {$categoryEntryId} as data identifier.");
        }

        if ($refId === null && $categoryEntryId === null) {
            $refId = $categoryEntry->create();
            $categoryEntry->createIdentifier($externalId, $refId);
            $logger->info("New category entry {$refId} created for external id {$externalId}.");
            $created = true;
        }

        if ($refId === null && $categoryEntryId) {
            $categoryEntry->createIdentifier($externalId, $categoryEntryId);
            $refId = $categoryEntryId;
            $logger->info("New external id {$externalId} reference created for category entry id {$refId}.");
        }

        return [
            $categoryEntry
                ->buildEntry($refId, $entry, $created)
        ];
    }

    /**
     * @param isys_cmdb_dao_category $dao
     * @param array                  $entries
     *
     * @return array
     */
    protected function prepareMultivalue(isys_cmdb_dao_category $dao, CiObject $ciObject, array $entries): array
    {
        $data = [];
        $externalType = "{$ciObject->getExtType()}/{$ciObject->getExtId()}/{$dao->get_category_const()}";
        $categoryEntry = new Category($ciObject, $dao, $externalType, $dao->get_category_const());
        $existingExternalIds = $categoryEntry->find();
        $logger = Logger::instance();

        foreach ($entries as $externalId => $entry) {
            $created = false;
            if (($refId = array_search($externalId, $existingExternalIds)) === false) {
                $refId = $categoryEntry->create();
                $categoryEntry->createIdentifier($externalId, $refId);
                $created = true;
                $logger->info("New category entry {$refId} and custom id {$externalId} reference created.");
            } else {
                $logger->info('Reference id for category found.');
            }

            $data[] = $categoryEntry->buildEntry($refId, $entry, $created);
        }

        return $data;
    }
}
