<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.com>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Property;

use isys_cmdb_dao_category;

/**
 * Class PropertyWriteEntityDefinition
 *
 * @package idoit\Module\Api\Property
 */
class PropertyWriteEntityDefinition
{
    private const SPECIFIC_DATA = [
        'isys_cmdb_dao_category_g_global::created'                                                => 'nonWriteable',
        'isys_cmdb_dao_category_g_global::created_by'                                             => 'nonWriteable',
        'isys_cmdb_dao_category_g_global::changed'                                                => 'nonWriteable',
        'isys_cmdb_dao_category_g_global::changed_by'                                             => 'nonWriteable',
        'isys_cmdb_dao_category_s_net::address_range'                                             => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_s_net::address_with_suffix'                                       => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_s_net::used_addresses'                                            => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_s_net::dns_server'                                                => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_s_net::dns_domain'                                                => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_s_net::layer2_assignments'                                        => 'nonWriteable', // @see API-554
        'isys_cmdb_dao_category_g_location::pos'                                                  => 'location_pos',
        'isys_cmdb_dao_category_g_network_port::interface'                                        => 'network_port_interface',
        'isys_cmdb_dao_category_g_network_port::assigned_connector'                               => 'network_port_assigned_connector',
        'isys_cmdb_dao_category_g_location::gps'                                                  => 'location_gps',
        'isys_cmdb_dao_category_g_location::latitude'                                             => 'location_lat_lng',
        'isys_cmdb_dao_category_g_location::longitude'                                            => 'location_lat_lng',
        'isys_cmdb_dao_category_g_ip::assigned_port'                                              => 'ip_assigned_port',
        'isys_cmdb_dao_category_g_ip::ipv4_address'                                               => 'ip_ipv4_address',
        'isys_cmdb_dao_category_s_layer2_net_assigned_logical_ports::isys_obj__id'                => 'layer2_net_assigned_logical_ports_object',
        'isys_cmdb_dao_category_s_layer2_net_assigned_logical_ports::isys_catg_log_port_list__id' => 'layer2_net_assigned_logical_ports_port',
        'isys_cmdb_dao_category_s_layer2_net::layer3_assignments'                                 => 'layer2_net_layer3_assignments',
    ];

    /**
     * @param isys_cmdb_dao_category $categoryDaoInstance
     * @param string                 $key
     * @param mixed                  $propertyDefinition
     *
     * @return PropertyWriteEntity
     */
    public static function get(isys_cmdb_dao_category $categoryDaoInstance, string $key, $propertyDefinition): PropertyWriteEntity
    {
        $daoClassName = get_class($categoryDaoInstance);

        if (isset(self::SPECIFIC_DATA["{$daoClassName}::{$key}"])) {
            $method = self::SPECIFIC_DATA["{$daoClassName}::{$key}"];

            return call_user_func("self::{$method}", $propertyDefinition);
        }

        switch ($propertyDefinition[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE]) {
            case C__PROPERTY__INFO__TYPE__DATE:
                return (new PropertyWriteEntity('Date', date('Y-m-d')))->addDefinition('string', 'ISO-8601 formatted date as string ("YYYY-MM-DD")', date('Y-m-d'));

            case C__PROPERTY__INFO__TYPE__DATETIME:
                return (new PropertyWriteEntity('Datetime', date('Y-m-d H:i:s')))
                    ->addDefinition('string', 'ISO-8601 formatted date time as string ("YYYY-MM-DD hh:mm:ss")', date('Y-m-d H:i:s'));

            case C__PROPERTY__INFO__TYPE__DIALOG:
                return (new PropertyWriteEntity('Dialog', 'Example Dialog title'))->addDefinition('integer', 'Id referencing an existing dialog entry.', 1)
                    ->addDefinition('string', 'String value that reference an existing dialog entry by its title or constant.', 'Dialog title #1');

            case C__PROPERTY__INFO__TYPE__DIALOG_PLUS:
                return (new PropertyWriteEntity('Dialog plus', 'Example Dialog Plus title'))->addDefinition('integer', 'Id referencing an existing dialog plus entry.', 1)
                    ->addDefinition(
                        'string',
                        'String value that reference an existing dialog entry by its title or constant. Otherwise a new entry will be created.',
                        'Dialog title #1'
                    );

            case C__PROPERTY__INFO__TYPE__DIALOG_LIST:
                return (new PropertyWriteEntity('Dialog list', ['List entry #1', 'List entry #2', 'List entry #3']))
                    ->addDefinition('integer[]', 'Array of ids referencing existing dialog entries.', [3, 4, 5])
                    ->addDefinition(
                        'string[]',
                        'Array of strings referencing existing dialog entries by title or constant. Unknown strings will result in new dialog entries.',
                        ['List entry #1', 'List entry #2', 'List entry #3']
                    );

            case C__PROPERTY__INFO__TYPE__DOUBLE:
                return (new PropertyWriteEntity('Double', 20000))->addDefinition('integer', 'A valid integer value', 20)
                    ->addDefinition('double', 'A valid double value.', 200.00);

            case C__PROPERTY__INFO__TYPE__FLOAT:
                return (new PropertyWriteEntity('Float', 100.00))->addDefinition('integer', 'A valid integer value', 10)
                    ->addDefinition('float', 'A valid float value.', 100.00);

            case C__PROPERTY__INFO__TYPE__INT:
                return (new PropertyWriteEntity('Integer', 10))->addDefinition('integer', 'A valid integer value', 10);

            case C__PROPERTY__INFO__TYPE__MONEY:
                return (new PropertyWriteEntity('Monetary', 5000.00))->addDefinition('integer', 'A valid integer value', 10)
                    ->addDefinition('float', 'A valid float value.', 100.00);

            case C__PROPERTY__INFO__TYPE__MULTISELECT:
                return (new PropertyWriteEntity('Multiselect', ['List entry #1', 'List entry #2', 'List entry #3']))
                    ->addDefinition('integer[]', 'Array of ids referencing existing dialog entries.', [1, 2, 3])
                    ->addDefinition(
                        'string',
                        'Array of strings referencing existing dialog entries by title or constant. Unknown strings will result in new dialog entries.',
                        ['List entry #1', 'List entry #2', 'List entry #3']
                    );

            case C__PROPERTY__INFO__TYPE__OBJECT_BROWSER:
                if ($propertyDefinition[C__PROPERTY__UI][C__PROPERTY__UI__PARAMS]['multiselection']) {
                    return (new PropertyWriteEntity('Object browser (multi)', [1, 2, 3]))
                        ->addDefinition('integer[]', 'Array of ids referencing existing cmdb objects.', [1, 2, 3]);
                }

                return (new PropertyWriteEntity('Object browser', 20))
                    ->addDefinition('integer', 'Id of an existing cmdb object or other entity (connector, license, etc...)', 20);

            case C__PROPERTY__INFO__TYPE__TEXT:
                return (new PropertyWriteEntity('Text', 'Example text value'))
                    ->addDefinition('string', 'Valid string value with maximum 255 characters.', 'Example text value');

            case C__PROPERTY__INFO__TYPE__TEXTAREA:
                return (new PropertyWriteEntity('Text area', 'Example text area value'))
                    ->addDefinition('string', 'Valid string value with maximum 65534 characters.', 'Example text area value');

            case C__PROPERTY__INFO__TYPE__UPLOAD:
                return (new PropertyWriteEntity('Upload', '__BASE_64_ENCODED_FILE__'))
                    ->addDefinition('string', 'String value representing a base64 encoded file', '__BASE_64_ENCODED_FILE__');

            case C__PROPERTY__INFO__TYPE__COMMENTARY:
                return (new PropertyWriteEntity('Commentary', 'Example commentary value'))
                    ->addDefinition('string', 'Valid string value with maximum 65534 characters', 'Example commentary value');
        }

        return new PropertyWriteEntity('Unknown Property', null);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    public static function network_port_interface($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Dialog', '1_C__CATG__HBA'))
            ->addDefinition(
                'string',
                "Strings that references an existing dialog entry via '<id>_<category const>'. Available category constants are 'C__CATG__HBA' and 'C__CATG__NETWORK_INTERFACE'",
                '<id>_<category const>'
            );
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    public static function network_port_assigned_connector($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Connection browser', 20))->addDefinition('integer', 'Id of an existing connector', 20);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    public static function location_pos($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Dialog', 10))->addDefinition('integer', 'Integer that references to position inside the rack', 10);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    public static function location_gps($property): PropertyWriteEntity
    {
        return new PropertyWriteEntity('Not writeable', null);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    public static function location_lat_lng($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Numeric', 12.345))
            ->addDefinition('string', 'A valid string that contains a numeric value.', '12.34')
            ->addDefinition('integer', 'A valid integer value', 12)
            ->addDefinition('float', 'A valid float value.', 12.345);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    private static function ip_assigned_port($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Dialog', '1_C__CATG__NETWORK_PORT'))
            ->addDefinition('string', 'Strings that references the title of an existing referenced port or logical port', 'Port 01')
            ->addDefinition(
                'string',
                "Strings that references an existing dialog entry via '<id>_<category const>'. Available category constants are 'C__CATG__NETWORK_PORT' and 'C__CATG__NETWORK_LOG_PORT'",
                '<id>_<category const>');
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    private static function layer2_net_assigned_logical_ports_object($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Object browser', 123))
            ->addDefinition('integer', 'ID of the layer 2 net object itself', 123);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    private static function layer2_net_assigned_logical_ports_port($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Port browser', [234]))
            ->addDefinition('integer[]', 'An array with a single integer representing the ID of the port, that should be attached to the layer 2 net', [234]);
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    private static function ip_ipv4_address($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('String', '127.0.0.1'))
            ->addDefinition('string', 'A string containing a valid IPv4 address', '127.0.0.1');
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     */
    private static function nonWriteable($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Empty', null));
    }

    /**
     * @param $property
     *
     * @return PropertyWriteEntity
     * @see API-470
     */
    private static function layer2_net_layer3_assignments($property): PropertyWriteEntity
    {
        return (new PropertyWriteEntity('Object browser (multi)', [20, 21]))
            ->addDefinition('integer[]', 'An array of integers representing the IDs of layer 3 nets, that should be attached to the layer 2 net', [20, 21]);
    }
}
