<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Request;

use idoit\Module\Api\Exception\JsonRpc\InternalErrorException;
use idoit\Module\Api\Exception\JsonRpc\MethodException;
use idoit\Module\Api\Exception\JsonRpc\ParameterException;
use idoit\Module\Api\Exception\JsonRpc\RequestException;
use idoit\Module\Api\Exception\JsonRpc\SystemException;

/**
 * Class RpcRequest
 *
 * @package idoit\Module\Api\Request
 */
class RpcRequest extends AbstractRequest
{

    /**
     * Is request a notification
     *
     * @return bool
     */
    public function isNotification()
    {
        return (array_key_exists('id', $this->getData()) === false);
    }

    /**
     * Validate request
     *
     * @return bool
     * @throws RequestException
     * @throws InternalErrorException
     */
    public function validate()
    {
        // Check for general validity
        if ($this->getVersion() !== '2.0' && empty($this->getMethod()) && (!array_key_exists('params', $this->getData()) || !is_array($this->getData()['params']))) {
            throw new RequestException('Invalid JSON-RPC request package');
        }

        // Invalid JSON-Version
        if ($this->getVersion() !== '2.0') {
            throw new RequestException('Invalid JSON-RPC Version. Use version 2.0');
        }

        // Invalid method
        if (!is_string($this->getMethod()) || $this->getMethod() === '') {
            throw new RequestException('Invalid JSON-RPC request package: Parameter "method" not found');
        }

        // Params not set
        if (!array_key_exists('params', $this->getData()) || !is_array($this->getData()['params'])) {
            throw new ParameterException('Invalid JSON-RPC request package: Parameter "params" not found.');
        }

        // Deprecated session parameter used
        if (isset($this->getData()['params']['session'])) {
            throw new SystemException('The session parameter is deprecated as of version 1.4. Use apikey and header authentication instead.');
        }

        // ApiKey is setted
        if (!is_string($this->getData()['params']['apikey']) || empty($this->getData()['params']['apikey'])) {
            throw new SystemException('Please provide a valid apiKey for request.');
        }

        return true;
    }

    /**
     * Get request id
     *
     * @return int
     */
    public function getId()
    {
        return $this->getData()['id'];
    }

    /**
     * Get request parameter
     *
     * @return array|null
     */
    public function getParams()
    {
        return $this->getData()['params'];
    }

    /**
     * Get request method
     *
     * @return string
     */
    public function getMethod()
    {
        return $this->getData()['method'];
    }

    /**
     * Get api key
     *
     * @return mixed
     */
    public function getApiKey()
    {
        return $this->getData()['params']['apikey'];
    }

    /**
     * Get request language
     *
     * @return string
     */
    public function getLanguage()
    {
        return $this->getData()['params']['language'] ?: 'en';
    }

    /**
     * Get request versiom
     *
     * @return mixed
     */
    public function getVersion()
    {
        return $this->getData()['jsonrpc'] ?: $this->getData()['version'];
    }
}
