<?php

/**
 * i-doit
 *
 * @package    i-doit
 * @subpackage API
 * @author     Selcuk Kekec <skekec@i-doit.de>
 * @version    1.10
 * @copyright  synetics GmbH
 * @license    http://www.i-doit.com/license
 */

namespace idoit\Module\Api\Validation;

use idoit\Component\Property\Property;
use idoit\Module\Api\Exception\ValidationException;
use idoit\Module\Api\Validation\Property\AutoText;
use idoit\Module\Api\Validation\Property\Commentary;
use idoit\Module\Api\Validation\Property\Date;
use idoit\Module\Api\Validation\Property\DateTime;
use idoit\Module\Api\Validation\Property\Dialog;
use idoit\Module\Api\Validation\Property\DialogList;
use idoit\Module\Api\Validation\Property\DialogPlus;
use idoit\Module\Api\Validation\Property\Double;
use idoit\Module\Api\Validation\Property\FloatValue as FLV;
use idoit\Module\Api\Validation\Property\Integer;
use idoit\Module\Api\Validation\Property\Money;
use idoit\Module\Api\Validation\Property\MultiSelect;
use idoit\Module\Api\Validation\Property\N2M;
use idoit\Module\Api\Validation\Property\ObjectBrowser;
use idoit\Module\Api\Validation\Property\Password;
use idoit\Module\Api\Validation\Property\Text;
use idoit\Module\Api\Validation\Property\TextArea;
use idoit\Module\Api\Validation\Property\Time;
use idoit\Module\Api\Validation\Property\Timeperiod;
use idoit\Module\Api\Validation\Property\Upload;

/**
 * Class Validation
 *
 * @package idoit\Module\Api\Validation
 */
abstract class PropertyValidationFactory
{
    /**
     * Validator registry
     *
     * @var array
     */
    const REGISTER = [
        C__PROPERTY__INFO__TYPE__AUTOTEXT       => AutoText::class,
        C__PROPERTY__INFO__TYPE__COMMENTARY     => Commentary::class,
        C__PROPERTY__INFO__TYPE__DATE           => Date::class,
        C__PROPERTY__INFO__TYPE__DATETIME       => DateTime::class,
        C__PROPERTY__INFO__TYPE__DIALOG         => Dialog::class,
        C__PROPERTY__INFO__TYPE__DIALOG_LIST    => DialogList::class,
        C__PROPERTY__INFO__TYPE__DIALOG_PLUS    => DialogPlus::class,
        C__PROPERTY__INFO__TYPE__DOUBLE         => Double::class,
        C__PROPERTY__INFO__TYPE__FLOAT          => FLV::class,
        C__PROPERTY__INFO__TYPE__INT            => Integer::class,
        C__PROPERTY__INFO__TYPE__MONEY          => Money::class,
        C__PROPERTY__INFO__TYPE__MULTISELECT    => MultiSelect::class,
        C__PROPERTY__INFO__TYPE__N2M            => N2M::class,
        C__PROPERTY__INFO__TYPE__OBJECT_BROWSER => ObjectBrowser::class,
        C__PROPERTY__INFO__TYPE__PASSWORD       => Password::class,
        C__PROPERTY__INFO__TYPE__TEXT           => Text::class,
        C__PROPERTY__INFO__TYPE__TEXTAREA       => TextArea::class,
        C__PROPERTY__INFO__TYPE__TIMEPERIOD     => Timeperiod::class,
        C__PROPERTY__INFO__TYPE__TIME           => Time::class,
        C__PROPERTY__INFO__TYPE__UPLOAD         => Upload::class,
    ];

    /**
     * Get validator
     *
     * @param                $title
     * @param                $value
     * @param array|Property $propertyDefinition
     *
     * @return PropertyValidation
     * @throws ValidationException
     */
    public static function getValidator($title, $value, $propertyDefinition)
    {
        if (!($propertyDefinition instanceof Property) && !is_array($propertyDefinition)) {
            throw new ValidationException('The provided property definition needs to be an array or instance of Property.');
        }

        // Check for emptiness.
        if (empty($propertyDefinition)) {
            /**
             * @todo Should this interrupt api call?
             */
            throw new ValidationException('unknown', 'Unable to retrieve property type. Please provide a valid definition.');
        }

        // @see  API-282  Virtual properties should not be accessible via API.
        if ($propertyDefinition[C__PROPERTY__PROVIDES][C__PROPERTY__PROVIDES__VIRTUAL] && $propertyDefinition[C__PROPERTY__PROVIDES][C__PROPERTY__PROVIDES__VALIDATION]) {
            throw new ValidationException('The provided property is virtual and can not be updated via API. Please provide a valid definition.');
        }

        // Get property type
        $propertyType = $propertyDefinition[C__PROPERTY__INFO][C__PROPERTY__INFO__TYPE];

        // Check whether corresponding validator is registered
        if (array_key_exists($propertyType, self::REGISTER) === false) {
            /**
             * @todo Should this interrupt api call?
             */
            throw new ValidationException($propertyType, 'Unable to retrieve corresponding validator for property type \'' . $propertyType . '\'');
        }

        // Check whether class is existing
        if (!class_exists(self::REGISTER[$propertyType])) {
            /**
             * @todo Should this interrupt api call?
             */
            throw new ValidationException($propertyType, 'Registered validation class does not exists.');
        }

        // Get validation class
        $validatorClass = self::REGISTER[$propertyType];

        // Create validation instance
        $validatorInstance = new $validatorClass($title, $value, $propertyDefinition);

        /** @var PropertyValidation $validatorInstance */
        return $validatorInstance;
    }
}
